/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.xml;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.w3c.dom.Element;

/**
 * a set of {@link Binding}s divided into categories managed by a {@link Parser}
 * .
 *
 * <a href="./Parser.html#bindings">See also 'Bindinds'</a>
 *
 * @see Parser
 * @author Tom Baeyens
 */
public class Bindings {

  /** maps categories to a list of bindings */
  protected Map<String, List<Binding>> bindings = null;

  public Bindings() {
  }

  /**
   * to be used when you want to customize the binding behaviour of a
   * {@link Parser}.
   */
  public Bindings(final Bindings other) {
    if (other.bindings != null) {
      this.bindings = new HashMap<String, List<Binding>>();
      final Set<String> categorieNames = other.bindings.keySet();
      if (categorieNames != null) {
        for (final String categoryName : categorieNames) {
          final List<Binding> categoryBindings = other.bindings.get(categoryName);
          final List<Binding> categoryBindingsCopy = new ArrayList<Binding>(
              categoryBindings);
          this.bindings.put(categoryName, categoryBindingsCopy);
        }
      }
    }
  }

  /**
   * get a binding for the given element and category. If the category is null,
   * then all the categories will be searched for a binding in random order.
   */
  public Binding getBinding(final Element element) {
    return this.getBinding(element, (String) null);
  }

  /**
   * get a binding for the given element and category. If the category is null,
   * then all the categories will be searched for a binding in random order.
   */
  public Binding getBinding(final Element element, final String category) {
    if (this.bindings == null) {
      return null;
    }
    // if no category is specified
    if (category == null) {
      // search in all categories
      for (final List<Binding> categoryBindings : this.bindings.values()) {
        final Binding binding = this.getBinding(element, categoryBindings);
        if (binding != null) {
          return binding;
        }
      }
    } else {
      final List<Binding> categoryBindings = this.bindings.get(category);
      if (categoryBindings != null) {
        return this.getBinding(element, categoryBindings);
      }
    }
    return null;
  }

  protected Binding getBinding(final Element element, final List<Binding> categoryBindings) {
    for (final Binding binding : categoryBindings) {
      if (binding.matches(element)) {
        return binding;
      }
    }
    return null;
  }

  /**
   * add an elementParser to this parser that will handle parsing of elements of
   * the given tagName for the default category.
   */
  public void addBinding(final Binding binding) {
    if (this.bindings == null) {
      this.bindings = new HashMap<String, List<Binding>>();
    }
    final String category = binding.getCategory();
    List<Binding> categoryBindings = this.bindings.get(category);
    if (categoryBindings == null) {
      categoryBindings = new ArrayList<Binding>();
      this.bindings.put(category, categoryBindings);
    }
    categoryBindings.add(binding);
  }

  /**
   * the set of all tagNames for which there is a binding specified in the given
   * category
   */
  public Set<String> getTagNames(final String category) {
    final List<Binding> categoryBindings = this.bindings != null ? this.bindings.get(category) : null;
    final Set<String> tagNames = new HashSet<String>();
    if (categoryBindings != null) {
      for (final Binding binding : categoryBindings) {
        tagNames.add(binding.toString());
      }
    }
    return tagNames;
  }

  /**
   * @return the bindings
   */
  public Map<String, List<Binding>> getBindings() {
    return this.bindings;
  }
}
