/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.model;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Stack;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.activity.Activity;
import org.ow2.orchestra.pvm.client.ClientProcessDefinition;
import org.ow2.orchestra.pvm.internal.model.CompositeElementImpl;
import org.ow2.orchestra.pvm.internal.model.EventImpl;
import org.ow2.orchestra.pvm.internal.model.EventListenerReference;
import org.ow2.orchestra.pvm.internal.model.ExceptionHandlerImpl;
import org.ow2.orchestra.pvm.internal.model.NodeImpl;
import org.ow2.orchestra.pvm.internal.model.ObservableElementImpl;
import org.ow2.orchestra.pvm.internal.model.ProcessDefinitionImpl;
import org.ow2.orchestra.pvm.internal.model.ProcessElementImpl;
import org.ow2.orchestra.pvm.internal.model.TimerDefinitionImpl;
import org.ow2.orchestra.pvm.internal.model.TransitionImpl;
import org.ow2.orchestra.pvm.internal.model.VariableDefinitionImpl;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ObjectDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ProvidedObjectDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.StringDescriptor;
import org.ow2.orchestra.pvm.listener.EventListener;

/**
 * factory for process definitions.
 *
 * <p>
 * Use this factory as a <a
 * href="http://martinfowler.com/bliki/FluentInterface.html">fluent
 * interface</a> for building a process definition. To use it in this way, start
 * with instantiating a ProcessFactory object. Then a number of methods can be
 * invoked concatenated with dots cause all the methods return the same process
 * factory object. When done, end that sequence with {@link #done()} to get the
 * constructed ProcessDefinition.
 * </p>
 *
 * <p>
 * The idea is that this results into a more compact and more readable code to
 * build process definitions as opposed to including xml inline. For example :
 * </p>
 *
 * <pre>
 * ProcessDefinition processDefinition = ProcessFactory.build().node().initial()
 *     .behaviour(new WaitState()).transition(&quot;normal&quot;).to(&quot;a&quot;).transition(
 *         &quot;shortcut&quot;).to(&quot;c&quot;).node(&quot;a&quot;).behaviour(new WaitState()).transition()
 *     .to(&quot;b&quot;).node(&quot;b&quot;).behaviour(new WaitState()).transition().to(&quot;c&quot;)
 *     .node(&quot;c&quot;).behaviour(new WaitState()).done();
 * </pre>
 *
 * <hr />
 *
 * <p>
 * If more control is needed over the creation of the process definition
 * objects, then consider using the concrete implementation classes from package
 * {@link org.ow2.orchestra.pvm.internal.model} directly. The implementation code
 * of this class might be a good guide to get you on your way.
 * </p>
 *
 * @author Tom Baeyens
 */
public class ProcessFactory {

  // static factory methods ///////////////////////////////////////////////////

  protected ProcessDefinitionImpl processDefinition;
  protected NodeImpl node;
  protected TransitionImpl transition;
  protected List<DestinationReference> destinationReferences;
  protected ObservableElementImpl observableElement;
  protected EventImpl event;
  protected EventListenerReference eventListenerReference;
  protected ExceptionHandlerImpl exceptionHandler;
  protected CompositeElementImpl compositeElement;
  protected CompositeElementImpl scope;
  protected Stack<CompositeElementImpl> compositeElementStack;

  /** start building a process definition without a name. */
  protected ProcessFactory() {
    this(null);
  }

  /** start building a process definition with the given name. */
  protected ProcessFactory(final String processName) {
    this(processName, null);
  }

  /** start building a process definition with the given name. */
  protected ProcessFactory(final String processName,
      final ProcessDefinitionImpl processDefinition) {
    if (processDefinition != null) {
      this.processDefinition = processDefinition;
    } else {
      this.processDefinition = this.instantiateProcessDefinition();
    }
    this.processDefinition.setName(processName);
    this.observableElement = this.processDefinition;
    this.compositeElement = this.processDefinition;
    this.scope = this.processDefinition;
  }

  /** starts building a process definition */
  public static ProcessFactory build() {
    return new ProcessFactory();
  }

  /** starts building a process definition */
  public static ProcessFactory build(final String processName) {
    return new ProcessFactory(processName);
  }

  /** starts populating a given process definition */
  public static ProcessFactory build(final String processName,
      final ProcessDefinitionImpl processDefinition) {
    return new ProcessFactory(processName, processDefinition);
  }

  /** to be overwritten by specific process language factories */
  protected ProcessDefinitionImpl instantiateProcessDefinition() {
    return new ProcessDefinitionImpl();
  }

  /** marks the last created node as the initial node in the process. */
  public ProcessFactory initial() {
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    if (this.processDefinition.getInitial() != null) {
      throw new PvmException("duplicate initial node");
    }
    this.processDefinition.setInitial(this.node);
    return this;
  }

  /**
   * applies on a node and makes it create a local activity instance scope. This
   * is automatically implied when {@link #variable(String) adding a variable} or
   * {@link #timer() adding a timer}
   */
  public ProcessFactory scope() {
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    this.node.setLocalScope(true);
    this.scope = this.node;
    return this;
  }

  /** declares a local variable. {@link #scope()} is automatically implied. */
  public ProcessFactory variable(final String key) {
    if (this.node != null) {
      this.scope();
    }
    final VariableDefinitionImpl variableDefinition = this.compositeElement
        .createVariableDefinition();
    variableDefinition.setKey(key);
    return this;
  }

  /** declares a local variable. {@link #scope()} is automatically implied. */
  public ProcessFactory variable(final Descriptor sourceDescriptor) {
    if (this.node != null && this.scope == null) {
      this.scope();
    }
    final VariableDefinitionImpl variableDefinition = this.scope
        .createVariableDefinition();
    variableDefinition.setKey(sourceDescriptor.getName());
    variableDefinition.setSourceDescriptor(sourceDescriptor);
    return this;
  }

  /** declares a local variable. {@link #scope()} is automatically implied. */
  public ProcessFactory variable(final String key, final String initialValue) {
    return this.variable(new StringDescriptor(key, initialValue));
  }

  /**
   * declares a timer on the current node or process. {@link #scope()} is
   * automatically implied.
   */
  public ProcessFactory timer(final String dueDateDescription, final String signalName) {
    return this.timer(dueDateDescription, null, signalName, null);
  }

  /**
   * declares a timer on the current node or process. {@link #scope()} is
   * automatically implied.
   */
  public ProcessFactory timer(final String dueDateDescription, final String signalName,
      final String repeat) {
    return this.timer(dueDateDescription, null, signalName, repeat);
  }

  /**
   * declares a timer on the current node or process. {@link #scope()} is
   * automatically implied.
   */
  public ProcessFactory timer(final Date dueDate, final String signalName) {
    return this.timer(null, dueDate, signalName, null);
  }

  protected ProcessFactory timer(final String dueDateDescription, final Date dueDate,
      final String signalName, final String repeat) {
    if (this.node != null && this.scope == null) {
      this.scope();
    }
    final TimerDefinitionImpl timerDefinition = this.scope.createTimerDefinition();
    if (dueDate != null) {
      timerDefinition.setDueDate(dueDate);
    } else {
      timerDefinition.setDueDateDescription(dueDateDescription);
    }
    timerDefinition.setSignalName(signalName);
    timerDefinition.setRepeat(repeat);
    return this;
  }

  /**
   * creates a node in the current parent. The current parent is either the
   * process definition or a composite node in case method
   * {@link #compositeNode(String)} was called previously.
   */
  public ProcessFactory node() {
    return this.node(null);
  }

  /**
   * creates a named node. The current parent is either the process definition
   * or a composite node in case method {@link #compositeNode(String)} was
   * called previously.
   */
  public ProcessFactory node(final String nodeName) {
    if (this.exceptionHandler != null) {
      this.exceptionHandler.setNodeName(nodeName);
    } else {
      this.node = this.compositeElement.createNode(nodeName);
      this.scope = null;

      this.observableElement = this.node;
      this.event = null;
      this.eventListenerReference = null;
      this.transition = null;
      this.exceptionHandler = null;
    }
    return this;
  }

  /**
   * sets the behaviour on the current node. A current node is required.
   */
  public ProcessFactory behaviour(final Activity activity) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    this.node.setBehaviour(activity);
    return this;
  }

  /**
   * sets the behaviour on the current node. A current node is required.
   */
  public ProcessFactory behaviour(final Descriptor descriptor) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    this.node.setBehaviour(descriptor);
    return this;
  }

  /**
   * sets the behaviour on the current node. A current node is required.
   */
  public ProcessFactory behaviour(final Class< ? extends Activity> activityClass) {
    return this.behaviour(new ObjectDescriptor(activityClass));
  }

  /**
   * sets the behaviour on the current node. A current node is required.
   */
  public ProcessFactory behaviour(final String expression) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    this.node.setBehaviour(expression);
    return this;
  }

  /**
   * sets the asyncExecute property on the current node. A current node is
   * required.
   */
  public ProcessFactory asyncExecute() {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    this.node.setExecutionAsync(true);
    return this;
  }

  /**
   * sets the asyncLeave property on the current node. A current node is
   * required.
   */
  public ProcessFactory asyncLeave() {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    this.node.setLeaveAsync(true);
    return this;
  }

  /**
   * sets the asyncSignal property on the current node. A current node is
   * required.
   */
  public ProcessFactory asyncSignal() {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    this.node.setSignalAsync(true);
    return this;
  }

  /**
   * sets the property needsPrevious on the current node. A current node is
   * required.
   */
  public ProcessFactory needsPrevious() {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.node == null) {
      throw new PvmException("no current node");
    }
    this.node.setPreviousNeeded(true);
    return this;
  }

  /**
   * starts a block in which nested nodes can be created. This block can be
   * ended with {@link #compositeEnd()}. A current node is required.
   */
  public ProcessFactory compositeNode() {
    return this.compositeNode(null);
  }

  /**
   * starts a block in which nested nodes can be created. This block can be
   * ended with {@link #compositeEnd()}. A current node is required.
   */
  public ProcessFactory compositeNode(final String nodeName) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }

    if (this.compositeElementStack == null) {
      this.compositeElementStack = new Stack<CompositeElementImpl>();
    }

    this.compositeElementStack.push(this.compositeElement);
    this.node(nodeName);
    this.compositeElement = this.node;

    return this;
  }

  /**
   * ends a block in which nested nodes are created. This method requires that a
   * nested node block was started before with {@link #compositeNode(String)}
   */
  public ProcessFactory compositeEnd() {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }

    if (this.compositeElementStack == null) {
      throw new PvmException("no composite node was started");
    }

    this.compositeElement = this.compositeElementStack.pop();

    if (this.compositeElementStack.isEmpty()) {
      this.compositeElementStack = null;
    }

    return this;
  }

  /**
   * creates a transition on the current node. This method requires a current
   * node
   */
  public ProcessFactory transition() {
    return this.transition(null);
  }

  /**
   * creates a named transition on the current node. This method requires a
   * current node
   */
  public ProcessFactory transition(final String transitionName) {
    if (this.exceptionHandler != null) {
      this.exceptionHandler.setTransitionName(transitionName);
    } else {
      if (this.node == null) {
        throw new PvmException("no current node");
      }
      this.transition = this.node.createOutgoingTransition(null, transitionName);
      this.observableElement = this.transition;
      this.event = null;
      this.eventListenerReference = null;
      this.exceptionHandler = null;
    }
    return this;
  }

  /**
   * sets the takeAsync property on the current transition This method requires
   * a current transition.
   */
  public ProcessFactory asyncTake() {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.transition == null) {
      throw new PvmException("no current transition");
    }
    this.transition.setTakeAsync(true);
    return this;
  }

  /**
   * sets the destination node on the current transition. This method requires a
   * current transition.
   */
  public ProcessFactory to(final String destination) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.transition == null) {
      throw new PvmException("no current transition");
    }
    if (this.destinationReferences == null) {
      this.destinationReferences = new ArrayList<DestinationReference>();
    }
    this.destinationReferences
        .add(new DestinationReference(this.transition, destination));
    return this;
  }

  /**
   * sets the wait condition on the current transition. This method requires a
   * current transition.
   */
  public ProcessFactory waitCondition(final Condition condition) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.transition == null) {
      throw new PvmException("no current transition");
    }
    final Descriptor conditionDescriptor = new ProvidedObjectDescriptor(condition);
    this.transition.setWaitConditionDescriptor(conditionDescriptor);
    return this;
  }

  /**
   * sets the guard condition on the current transition. This method requires a
   * current transition.
   */
  public ProcessFactory guardCondition(final Condition condition) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.transition == null) {
      throw new PvmException("no current transition");
    }
    final Descriptor conditionDescriptor = new ProvidedObjectDescriptor(condition);
    this.transition.setConditionDescriptor(conditionDescriptor);
    return this;
  }

  /**
   * creates the given event on the current process element. This method
   * requires a process element. A process element is either a process
   * definition or a node. This method doesn't need to be called for
   * transitions. If you have exception handlers and listeners on an event, make
   * sure that you put the invocations of {@link #exceptionHandler(Class)}
   * first.
   */
  public ProcessFactory event(final String eventName) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.observableElement == null) {
      throw new PvmException("no current process element");
    }
    if (this.observableElement instanceof Transition) {
      throw new PvmException(
          "for actions on transitions, you don't need to call event");
    }
    this.event = this.observableElement.createEvent(eventName);
    this.exceptionHandler = null;
    return this;
  }

  /**
   * creates an exception handler for the given exception class on the current
   * process element; until the {@link #exceptionHandlerEnd()}. Subsequent
   * invocations of {@link #listener(Activity) listeners} or
   * {@link #transition() transitions} will have the created exception handler as
   * a target.
   *
   * DONT'T FORGET TO CLOSE THE EXCEPTION HANDLER WITH exceptionHandlerEnd.
   */
  public ProcessFactory exceptionHandler(
      final Class< ? extends Exception> exceptionClass) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }

    ProcessElementImpl processElement = null;
    if (this.eventListenerReference != null) {
      processElement = this.eventListenerReference;
    } else if (this.event != null) {
      processElement = this.event;
    } else if (this.observableElement != null) {
      processElement = this.observableElement;
    } else {
      throw new PvmException("no current process element, event or action");
    }

    this.exceptionHandler = processElement.createExceptionHandler();

    if (exceptionClass != null) {
      this.exceptionHandler.setExceptionClassName(exceptionClass.getName());
    }

    return this;
  }

  public ProcessFactory exceptionHandlerEnd() {
    this.exceptionHandler = null;
    return this;
  }

  public ProcessFactory transactional() {
    if (this.exceptionHandler == null) {
      throw new PvmException(
          "transactional is a property of an exception handler");
    }
    this.exceptionHandler.setTransactional(true);
    return this;
  }

  /**
   * adds an action to the current event. The current event was either created
   * by {@link #event(String)} or by a {@link #transition()}. Subsequent
   * invocations of {@link #exceptionHandler(Class)} will be associated to this
   * event listener.
   */
  public ProcessFactory listener(final Descriptor descriptor) {
    if (this.exceptionHandler != null) {
      this.exceptionHandler.createEventListenerReference(descriptor);
    } else {
      this.getEvent().createEventListenerReference(descriptor);
    }
    return this;
  }

  /**
   * adds an action to the current event. The current event was either created
   * by {@link #event(String)} or by a {@link #transition()}. Subsequent
   * invocations of {@link #exceptionHandler(Class)} will be associated to this
   * event listener.
   */
  public ProcessFactory listener(final EventListener eventListener) {
    if (this.exceptionHandler != null) {
      this.exceptionHandler.createEventListenerReference(eventListener);
    } else {
      this.eventListenerReference = this.getEvent().createEventListenerReference(
          eventListener);
    }
    return this;
  }

  /**
   * adds an action to the current event. The current event was either created
   * by {@link #event(String)} or by a {@link #transition()}. Subsequent
   * invocations of {@link #exceptionHandler(Class)} will be associated to this
   * event listener.
   */
  public ProcessFactory listener(final String expression) {
    if (this.exceptionHandler != null) {
      this.exceptionHandler.createActivityReference(expression);
    } else {
      this.eventListenerReference = this.getEvent().createEventListenerReference(
          expression);
    }
    return this;
  }

  /**
   * disables propagated events. This means that this action will only be
   * executed if the event is fired on the actual process element of the event.
   * The current action will not be executed if an event is fired on one of the
   * children of the process element to which this event relates.
   */
  public ProcessFactory propagationDisabled() {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.eventListenerReference == null) {
      throw new PvmException("no current event action");
    }
    this.eventListenerReference.setPropagationEnabled(false);
    return this;
  }

  private EventImpl getEvent() {
    if ((this.event == null) && (this.observableElement instanceof TransitionImpl)) {
      this.event = ((TransitionImpl) this.observableElement).createEvent();
      return this.event;
    }
    if (this.event == null) {
      throw new PvmException("no current event");
    }
    return this.event;
  }

  /** adds a string-valued configuration to the current process element */
  public ProcessFactory property(final String name, final String stringValue) {
    final StringDescriptor stringDescriptor = new StringDescriptor();
    stringDescriptor.setName(name);
    stringDescriptor.setValue(stringValue);
    this.property(stringDescriptor);
    return this;
  }

  /** adds a configuration to the current process element */
  public ProcessFactory property(final Descriptor descriptor) {
    if (this.exceptionHandler != null) {
      throw new PvmException(
          "exceptionHandler needs to be closed with exceptionHandlerEnd");
    }
    if (this.observableElement == null) {
      throw new PvmException("no current process element");
    }
    if (this.event != null) {
      this.event.addProperty(descriptor);
    } else {
      this.observableElement.addProperty(descriptor);
    }
    return this;
  }

  public class DestinationReference {
    private final TransitionImpl transition;
    private final String destinationName;

    public DestinationReference(final TransitionImpl transition,
        final String destinationName) {
      this.transition = transition;
      this.destinationName = destinationName;
    }

    public void resolve() {
      final NodeImpl destination = (NodeImpl) ProcessFactory.this.processDefinition
          .findNode(this.destinationName);
      if (destination == null) {
        throw new PvmException("couldn't find destination node '"
            + this.destinationName + "' for transition " + this.transition);
      }
      destination.addIncomingTransition(this.transition);
      this.transition.setDestination(destination);
    }
  }

  /**
   * extract the process definition from the factory. This should be the last
   * method in the chain of subsequent invoked methods on this factory object.
   */
  public ClientProcessDefinition done() {
    this.resolveDestinations();
    if (this.processDefinition.getInitial() == null) {
      throw new PvmException("no initial node");
    }
    return this.processDefinition;
  }

  /**
   * sets the {@link org.ow2.orchestra.pvm.ProcessDefinition#getVersion() version} of the process
   * definition explicitely
   */
  public ProcessFactory version(final int version) {
    this.processDefinition.setVersion(version);
    return this;
  }

  /**
   * sets the {@link org.ow2.orchestra.pvm.ProcessDefinition#getKey() key} of the process definition
   * explicitely
   */
  public ProcessFactory key(final String key) {
    this.processDefinition.setKey(key);
    return this;
  }

  private void resolveDestinations() {
    if (this.destinationReferences != null) {
      for (final DestinationReference destinationReference : this.destinationReferences) {
        destinationReference.resolve();
      }
    }
  }
}
