/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.processlog;

import java.text.SimpleDateFormat;
import java.util.List;

import org.ow2.orchestra.pvm.Execution;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * @author Tom Baeyens
 */
public class ProcessLogXmlSerializer {

  public static final SimpleDateFormat dateFormatter = new SimpleDateFormat(
      "dd/MM/yyyy HH:mm:ss,SSS");

  protected List<ProcessLogProperty> properties;

  public void appendLog(final Document document, final Element parent, final ProcessLog processLog) {
    final String logType = this.getEventType(processLog);
    final String time = this.getTime(processLog);
    final String executionDbid = this.getExecutionDbid(processLog);
    final String processInstanceDbid = this.getProcessInstanceDbid(processLog);
    final String processInstanceKey = this.getProcessInstanceKey(processLog);

    final Element logElement = document.createElement("log");
    parent.appendChild(logElement);

    logElement.setAttribute("type", logType);
    logElement.setAttribute("time", time);
    if (executionDbid != null) {
      logElement.setAttribute("execution", executionDbid);
    }
    if (processInstanceDbid != null) {
      logElement.setAttribute("instance", processInstanceDbid);
    }
    if (processInstanceKey != null) {
      logElement.setAttribute("key", processInstanceKey);
    }

    this.appendProperties(document, logElement, processLog.getProperties());
  }

  protected void appendProperties(final Document document, final Element parent,
      final List<ProcessLogProperty> properties) {
    if (properties != null) {
      for (final ProcessLogProperty property : properties) {
        final Element propertyElement = document.createElement("property");
        parent.appendChild(propertyElement);

        if (property.name != null) {
          propertyElement.setAttribute("name", property.name);
        }
        if (property.value != null) {
          propertyElement.setAttribute("value", property.value);
        }
        this.appendProperties(document, propertyElement, property.getProperties());
      }
    }
  }

  protected String getEventType(final ProcessLog processLog) {
    return processLog.getType();
  }

  protected String getTime(final ProcessLog processLog) {
    return ProcessLogXmlSerializer.dateFormatter.format(processLog.getTime());
  }

  protected String getExecutionDbid(final ProcessLog processLog) {
    final Execution execution = processLog.getExecution();
    return execution != null ? Long.toString(execution.getDbid()) : null;
  }

  protected String getProcessInstanceDbid(final ProcessLog processLog) {
    final Execution processInstance = processLog.getProcessInstance();
    return processInstance != null ? Long.toString(processInstance.getDbid()) : null;
  }

  protected String getProcessInstanceKey(final ProcessLog processLog) {
    final Execution processInstance = processLog.getProcessInstance();
    return processInstance != null ? processInstance.getKey() : null;
  }
}
