/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.env;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Set;

import org.ow2.orchestra.pvm.internal.env.PvmEnvironment;
import org.ow2.orchestra.pvm.internal.env.PvmEnvironmentFactoryParser;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.stream.FileStreamSource;
import org.ow2.orchestra.pvm.internal.stream.InputStreamSource;
import org.ow2.orchestra.pvm.internal.stream.ResourceStreamSource;
import org.ow2.orchestra.pvm.internal.stream.StreamSource;
import org.ow2.orchestra.pvm.internal.stream.StringStreamSource;
import org.ow2.orchestra.pvm.internal.stream.UrlStreamSource;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireDefinition;

/**
 * an environment factory that also is the environment-factory context.
 *
 * <p>
 * This environment factory will produce environments with 2 contexts: the
 * environment-factory context and the block context.
 * </p>
 *
 * <p>
 * An environment-factory context is build from two wire definitions: the
 * environment-factory wire definition and the environment wire definition.
 * </p>
 *
 * <p>
 * The environment-factory context itself is build from the environment-factory
 * wire definition. So all objects that are created in this context remain
 * cached for the lifetime of this environment-factory context object.
 * </p>
 *
 * <p>
 * This environment-factory context is also a environment factory. The produced
 * environments contain 2 contexts: the environment-factory context itself and a
 * new environment context, build from the environment wire definition. For each
 * created environment, a new environment context will be created from the same
 * environment wire definition. Objects in the environment context will live for
 * as long as the environment.
 * </p>
 *
 * @author Tom Baeyens
 */
public class PvmEnvironmentFactory extends EnvironmentFactory implements
    Context {

  private static final long serialVersionUID = 1L;
  private static final Log LOG = Log.getLog(PvmEnvironmentFactory.class.getName());
  protected static PvmEnvironmentFactoryParser pvmEnvironmentFactoryParser = new PvmEnvironmentFactoryParser();

  protected WireContext environmentFactoryCtxWireContext = null;
  protected WireDefinition environmentCtxWireDefinition = null;

  public PvmEnvironmentFactory() {
  }

  public PvmEnvironmentFactory(final String resource) {
    this(resource, null);
  }

  public PvmEnvironmentFactory(final String resource, final ClassLoader classLoader) {
    this(new ResourceStreamSource(resource, classLoader));
  }

  public PvmEnvironmentFactory(final File file) {
    this(new FileStreamSource(file));
  }

  public PvmEnvironmentFactory(final URL url) {
    this(new UrlStreamSource(url));
  }

  public PvmEnvironmentFactory(final InputStream inputStream) {
    this(new InputStreamSource(inputStream));
  }

  public static PvmEnvironmentFactory parseXmlString(final String xmlString) {
    return new PvmEnvironmentFactory(new StringStreamSource(xmlString));
  }

  PvmEnvironmentFactory(final StreamSource streamSource) {
    PvmEnvironmentFactoryParser.getInstance().createParse().setDocumentObject(
        this).setStreamSource(streamSource).execute().checkProblems(
        "pvm environment configuration " + streamSource);
  }

  @Override
  public Environment openEnvironment() {
    final PvmEnvironment environment = new PvmEnvironment(this);

    PvmEnvironmentFactory.LOG.debug("opening " + environment);

    // set the classloader
    final ClassLoader classLoader = this.environmentFactoryCtxWireContext.getClassLoader();
    if (classLoader != null) {
      environment.setClassLoader(classLoader);
    }

    // add the environment-factory context
    environment.addContext(this.environmentFactoryCtxWireContext);

    // add the environment block context
    final WireContext environmentContext = new WireContext(
        this.environmentCtxWireDefinition, Context.CONTEXTNAME_ENVIRONMENT,
        environment, true);
    // add the environment block context to the environment
    environment.addContext(environmentContext);

    try {
      // finish the creation of the environment wire context
      environmentContext.create();

    } catch (final RuntimeException e) {
      // On exception, pop environment
      Environment.popEnvironment();
      throw e;
    }
    // if all went well, return the created environment
    return environment;
  }

  @Override
  public void close() {
    this.environmentFactoryCtxWireContext.fire(WireContext.EVENT_CLOSE, null);
  }

  // environment-factory context delegation methods
  // ///////////////////////////////////

  public Object get(final String key) {
    return this.environmentFactoryCtxWireContext.get(key);
  }

  public <T> T get(final Class<T> type) {
    return this.environmentFactoryCtxWireContext.get(type);
  }

  public String getName() {
    return this.environmentFactoryCtxWireContext.getName();
  }

  public boolean has(final String key) {
    return this.environmentFactoryCtxWireContext.has(key);
  }

  public Set<String> keys() {
    return this.environmentFactoryCtxWireContext.keys();
  }

  public Object set(final String key, final Object value) {
    return this.environmentFactoryCtxWireContext.set(key, value);
  }

  // getters and setters //////////////////////////////////////////////////////

  public void setEnvironmentCtxWireDefinition(final WireDefinition blockWireDefinition) {
    this.environmentCtxWireDefinition = blockWireDefinition;
  }

  public WireContext getEnvironmentFactoryCtxWireContext() {
    return this.environmentFactoryCtxWireContext;
  }

  public void setEnvironmentFactoryCtxWireContext(
      final WireContext applicationWireContext) {
    this.environmentFactoryCtxWireContext = applicationWireContext;
  }

  public WireDefinition getEnvironmentCtxWireDefinition() {
    return this.environmentCtxWireDefinition;
  }

}
