/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor;

import java.util.Collection;
import java.util.Date;

import org.hibernate.StaleStateException;
import org.hibernate.exception.GenericJDBCException;
import org.hibernate.exception.LockAcquisitionException;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.internal.job.JobImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.svc.RetryInterceptor;

/**
 * Common part of dispatcher thread implementations.
 * This thread is responsible for acquiring jobs in the jobImpl that need to be
 * executed and to give them to the jobExecutor thread pool.
 * There is only one dispatcher thread per JobExecutor.
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher
 *
 */
public abstract class AbstractDispatcherThread extends Thread {

  private boolean isActive = true;
  private boolean checkForNewJobs;
  private final Object semaphore = new Object();

  private static final Log LOG = Log.getLog(AbstractDispatcherThread.class.getName());

  /**
   *
   */
  public AbstractDispatcherThread() {
    super();
  }

  /**
   * @param target
   */
  public AbstractDispatcherThread(final Runnable target) {
    super(target);
  }

  /**
   * @param name
   */
  public AbstractDispatcherThread(final String name) {
    super(name);
  }

  /**
   * @param group
   * @param target
   */
  public AbstractDispatcherThread(final ThreadGroup group, final Runnable target) {
    super(group, target);
  }

  protected abstract void putAcquiredJobDbidsOnQueue(Collection<JobImpl<?>> acquiredJobDbids);

  /**
   * @param group
   * @param name
   */
  public AbstractDispatcherThread(final ThreadGroup group, final String name) {
    super(group, name);
  }

  /**
   * @param target
   * @param name
   */
  public AbstractDispatcherThread(final Runnable target, final String name) {
    super(target, name);
  }

  /**
   * @param group
   * @param target
   * @param name
   */
  public AbstractDispatcherThread(final ThreadGroup group, final Runnable target, final String name) {
    super(group, target, name);
  }

  /**
   * @param group
   * @param target
   * @param name
   * @param stackSize
   */
  public AbstractDispatcherThread(final ThreadGroup group, final Runnable target, final String name, final long stackSize) {
    super(group, target, name, stackSize);
  }

  @Override
  public void run() {
    AbstractDispatcherThread.LOG.info("starting...");
    long delay = 50;
    int delayFactor = 2;
    final CommandService commandService = this.getJobExecutor().getCommandExecutor();
    if (commandService instanceof RetryInterceptor) {
      delay = ((RetryInterceptor) commandService).getDelay();
      delayFactor = ((RetryInterceptor) commandService).getDelayFactor();
    }
    long currentDelay = delay;
    try {
      while (this.isActive) {
        try {
          // checkForNewJobs is set to true in jobWasAdded() below
          this.checkForNewJobs = false;

          // try to acquire jobs
          final Collection<Collection<JobImpl<?>>> acquiredJobDbids = this.acquireJobs();

          // no exception so resetting the currentIdleInterval
          currentDelay = delay;
          if ((acquiredJobDbids != null) && (!acquiredJobDbids.isEmpty())) {
            for (final Collection<JobImpl<?>> jobs : acquiredJobDbids) {
              this.putAcquiredJobDbidsOnQueue(jobs);
              AbstractDispatcherThread.LOG.info("added jobs " + jobs + " to the queue");
            }

          } else if (this.isActive) {
            final long waitPeriod = this.getWaitPeriod();
            if (waitPeriod > 0) {
              synchronized (this.semaphore) {
                if (!this.checkForNewJobs) {
                  AbstractDispatcherThread.LOG.debug(this.getName() + " will wait for max " + waitPeriod
                      + "ms on " + this.getJobExecutor());
                  this.semaphore.wait(waitPeriod);
                  AbstractDispatcherThread.LOG.debug(this.getName() + " woke up");
                } else {
                  AbstractDispatcherThread.LOG.debug("skipped wait because new message arrived");
                }
              }
            }
          }

        } catch (final InterruptedException e) {
          AbstractDispatcherThread.LOG.info((this.isActive ? "active" : "inactivated")
              + " jobImpl dispatcher thread '" + this.getName()
              + "' got interrupted");
        } catch (final Exception e) {

          final String logMessage = "exception in job executor dispatcher thread."
            + " Waiting " + currentDelay + " milliseconds";
          if (e instanceof StaleStateException
              || e instanceof LockAcquisitionException
              || e instanceof GenericJDBCException) {
            AbstractDispatcherThread.LOG.trace(logMessage, e);
          } else {
            AbstractDispatcherThread.LOG.error(logMessage, e);
          }
          try {
            synchronized (this.semaphore) {
              this.semaphore.wait(currentDelay);
            }
          } catch (final InterruptedException e2) {
            AbstractDispatcherThread.LOG.trace("delay after exception got interrupted", e2);
          }
          // after an exception, the current idle interval is doubled to prevent
          // continuous exception generation when e.g. the db is unreachable
          currentDelay = currentDelay * delayFactor;
        }
      }
    } catch (final Throwable t) {
      t.printStackTrace();
    } finally {
      AbstractDispatcherThread.LOG.info(this.getName() + " leaves cyberspace");
    }
  }

  protected Collection<Collection<JobImpl<?>>> acquireJobs() {
    final CommandService commandService = this.getJobExecutor().getCommandExecutor();
    final Command<Collection<Collection<JobImpl<?>>>> acquireJobsCommand = this.getJobExecutor().getAcquireJobsCommand();
    return commandService.execute(acquireJobsCommand);
  }

  protected Date getNextDueDate() {
    final CommandService commandService = this.getJobExecutor().getCommandExecutor();
    final Command<Date> getNextDueDate = this.getJobExecutor().getNextDueDateCommand();
    return commandService.execute(getNextDueDate);
  }

  protected long getWaitPeriod() {
    long interval = this.getJobExecutor().getIdleMillis();
    final Date nextDueDate = this.getNextDueDate();
    if (nextDueDate != null) {
      final long currentTimeMillis = System.currentTimeMillis();
      final long nextDueDateTime = nextDueDate.getTime();
      if (nextDueDateTime < currentTimeMillis + interval) {
        interval = nextDueDateTime - currentTimeMillis;
      }
    }
    if (interval < 0) {
      interval = 0;
    }
    return interval;
  }

  public void deactivate() {
    this.deactivate(false);
  }

  public void deactivate(final boolean join) {
    if (this.isActive) {
      AbstractDispatcherThread.LOG.debug("deactivating " + this.getName());
      this.isActive = false;
      this.interrupt();
      if (join) {
        while (this.isAlive()) {
          try {
            AbstractDispatcherThread.LOG.debug("joining " + this.getName());
            this.join();
          } catch (final InterruptedException e) {
            AbstractDispatcherThread.LOG.trace("joining " + this.getName() + " got interrupted");
          }
        }
      }
    } else {
      AbstractDispatcherThread.LOG.trace("ignoring deactivate: " + this.getName() + " is not active");
    }
  }

  public void jobWasAdded() {
    AbstractDispatcherThread.LOG.trace("notifying jobImpl executor dispatcher thread of new jobImpl");
    synchronized (this.semaphore) {
      this.checkForNewJobs = true;
      this.semaphore.notify();
    }
  }

  public boolean isActive() {
    return this.isActive;
  }

  public abstract AbstractJobExecutor getJobExecutor();
}