/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor;

import java.util.Date;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.job.JobImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.job.Job;

/**
 * @author Tom Baeyens
 */
public class ExecuteJobCmd implements Command<Job> {

  private static final long serialVersionUID = 1L;

  private static final Log LOG = Log.getLog(ExecuteJobCmd.class.getName());

  protected JobImpl<?> jobDbid;

  public ExecuteJobCmd(final JobImpl<?> jobDbid) {
    this.jobDbid = jobDbid;
  }

  public Job execute(final Environment environment) {
    final JobDbSession session = environment.get(JobDbSession.class);
    if (session == null) {
      throw new PvmException("no job session configured to get job " + this.jobDbid
          + " for execution");
    }

    final int oldVersion = this.jobDbid.getDbversion();
    final JobImpl< ? > job = session.get(JobImpl.class, this.jobDbid.getDbid());

    if (job == null) {
      ExecuteJobCmd.LOG.debug("job " + this.jobDbid + " no longer exists");
      return null;
    }
    if (oldVersion != job.getDbversion()) {
      ExecuteJobCmd.LOG.debug("job " + this.jobDbid + " updated externally");
      return job;
    }
    // in case of exclusive jobs, the jobImpl might have been deleted
    // before we execute it (they are in a list)
    ExecuteJobCmd.LOG.debug("executing job " + job + "...");
    job.execute(environment);
    ExecuteJobCmd.LOG.debug("executed job " + job);

    // if this job is locked too long, it could be unlocked by the
    // lockmonitor and
    // executed by another thread.
    final Date lockExpirationDate = job.getLockExpirationTime();
    // can be null if it was rescheduled
    if (lockExpirationDate != null) {
      final long lockExpiration = lockExpirationDate.getTime();
      final long currentTime = System.currentTimeMillis();
      if (currentTime > lockExpiration) {
        throw new PvmException("job took too long: lock expired "
            + (currentTime - lockExpiration) + "ms ago");
      }
    }
    return job;
  }

}
