/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor.pvm;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.internal.jobexecutor.AbstractJobExecutor;
import org.ow2.orchestra.pvm.internal.jobexecutor.AcquireJobsCmd;
import org.ow2.orchestra.pvm.internal.jobexecutor.GetNextDueDateCmd;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobExecutor;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobHistoryEntry;
import org.ow2.orchestra.util.PrefixThreadFactory;

/**
 * manager for jobImpl execution threads and their configuration.
 *
 * @author Tom Baeyens, Guillaume Porcher
 */
public class PvmJobExecutor extends AbstractJobExecutor implements JobExecutor {

  private int nbrOfThreads = 3;
  private int historySize = 200;
  private ThreadPoolExecutor threadPool;

  private DispatcherThread dispatcherThread = null;

  /**
   * queue to dispatch collections of jobDbids to the JobExecutorThreads, which
   * are competing readers.
   */
  private LinkedBlockingQueue<Runnable> jobTasksQueue = null;

  private final List<JobHistoryEntry> history = new ArrayList<JobHistoryEntry>();

  /* (non-Javadoc)
   * @see org.ow2.orchestra.pvm.internal.jobexecutor.IJobExecutor#start()
   */
  @Override
  public synchronized void start() {
    if (this.commandService == null) {
      throw new PvmException(
          "no command executor available in jobImpl executor");
    }
    if (!this.isActive) {
      this.acquireJobsCommand = new AcquireJobsCmd(this, this.getName(), this.getLockMillis(), this.isLimitSameInstanceJobs());
      this.nextDueDateCommand = new GetNextDueDateCmd();

      // the max capacity of the jobDbidsQueue is set to
      // nbrOfJobExecutorThreads.
      // That way, the dispatcher thread will be stalled if enough jobs are
      // acquired.
      this.jobTasksQueue = new LinkedBlockingQueue<Runnable>(2 * this.nbrOfThreads);

      this.isActive = true;
      AbstractJobExecutor.LOG.trace("starting jobImpl executor threads for jobImpl executor '"
          + this.name + "'...");
      this.threadPool = new ThreadPoolExecutor(
          this.nbrOfThreads,
          this.nbrOfThreads,
          60L,
          TimeUnit.SECONDS,
          this.jobTasksQueue,
          new PrefixThreadFactory("JobExecutor"));

      try {
        final Method m = ThreadPoolExecutor.class.getDeclaredMethod("allowCoreThreadTimeOut", boolean.class);
        m.invoke(this.threadPool, true);
      } catch (final Exception e) {
        // method is only available in java 6
      }

      AbstractJobExecutor.LOG.trace("starting dispatcher thread for jobImpl executor '" + this.name
          + "'...");
      this.dispatcherThread = new DispatcherThread(this);
      this.dispatcherThread.start();

    } else {
      AbstractJobExecutor.LOG.trace("ignoring start: jobImpl executor '" + this.name
          + "' is already started'");
    }
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.pvm.internal.jobexecutor.IJobExecutor#stop()
   */
  @Override
  public synchronized void stop() {
    this.stop(false);
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.pvm.internal.jobexecutor.IJobExecutor#stop(boolean)
   */
  @Override
  public synchronized void stop(final boolean join) {
    AbstractJobExecutor.LOG.debug("stopping jobImpl executor");
    if (this.isActive) {
      this.isActive = false;
      this.dispatcherThread.deactivate(true);
      this.waitTillQueueEmpty();
      this.threadPool.shutdown();
      if (join) {
        try {
          this.threadPool.awaitTermination(180, TimeUnit.SECONDS);
        } catch (final InterruptedException e) {
        }
      }
    } else {
      AbstractJobExecutor.LOG.trace("ignoring stop: jobImpl executor '" + this.name + "' not started");
    }
  }

  protected void waitTillQueueEmpty() {
    while (!this.jobTasksQueue.isEmpty()) {
      AbstractJobExecutor.LOG.trace("waiting for jobImpl-id-queue to become empty");
      try {
        Thread.sleep(200);
      } catch (final InterruptedException e) {
        AbstractJobExecutor.LOG
            .trace("waiting for jobImpl-id-queue to become empty got interrupted");
      }
    }
  }

  protected BlockingQueue<Runnable> getJobTasksQueue() {
    return this.jobTasksQueue;
  }

  @Override
  protected int getAcquireJobSize() {
    return 1;
  }
  // getters //////////////////////////////////////////////////////////////////

  public int getHistorySize() {
    return this.historySize;
  }

  public int getNbrOfThreads() {
    return this.nbrOfThreads;
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.pvm.internal.jobexecutor.IJobExecutor#getDispatcherThread()
   */
  @Override
  public DispatcherThread getDispatcherThread() {
    return this.dispatcherThread;
  }

  /**
   * @return the threadPool
   */
  public ThreadPoolExecutor getThreadPool() {
    return this.threadPool;
  }

  public List<JobHistoryEntry> getHistory() {
    return this.history;
  }

  // configuration setters ////////////////////////////////////////////////////

  public void setNbrOfJobExecutorThreads(final int nbrOfJobExecutorThreads) {
    this.nbrOfThreads = nbrOfJobExecutorThreads;
  }

  public void setHistoryMaxSize(final int historyMaxSize) {
    this.historySize = historyMaxSize;
  }
}
