/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.tx;

import org.hibernate.Session;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * @author Tom Baeyens
 */
public class HibernateSessionResource implements StandardResource {

  private static final Log LOG = Log.getLog(HibernateSessionResource.class
      .getName());

  protected Session session;
  protected org.hibernate.Transaction transaction;

  public HibernateSessionResource(final Session session) {
    this.session = session;
    if (HibernateSessionResource.LOG.isTraceEnabled()) {
      HibernateSessionResource.LOG.trace("beginning transaction on hibernate session "
          + System.identityHashCode(session));
    }
    try {
      this.transaction = session.beginTransaction();
    } catch (final RuntimeException e) {
      HibernateSessionResource.LOG.error(
          "hibernate transaction begin failed.  closing hibernate session", e);
      session.close();
      throw e;
    }

    HibernateSessionResource.LOG.debug("begun hibernate transaction "
        + System.identityHashCode(this.transaction) + " on hibernate session "
        + System.identityHashCode(session));
  }

  public void prepare() {
    HibernateSessionResource.LOG.debug("flushing hibernate session " + System.identityHashCode(this.session));
    this.session.flush();
  }

  public void commit() {
    HibernateSessionResource.LOG.debug("committing hibernate transaction "
        + System.identityHashCode(this.transaction));
    try {
      this.transaction.commit();
    } finally {
      this.closeSession();
    }
  }

  private void closeSession() {
    HibernateSessionResource.LOG.debug("closing hibernate session " + System.identityHashCode(this.session));
    this.session.close();
  }

  public void rollback() {
    HibernateSessionResource.LOG.debug("rolling back hibernate transaction "
        + System.identityHashCode(this.transaction));
    try {
      this.transaction.rollback();
    } finally {
      this.closeSession();
    }
  }
}
