package org.ow2.orchestra.pvm.internal.util;

import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.Enumeration;
import java.util.List;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ArgDescriptor;

public abstract class ReflectUtil {

  private static Log log = Log.getLog(ReflectUtil.class.getName());

  private ReflectUtil() { }

  static ClassLoader resolveClassLoader(ClassLoader classLoader) {
    // 1) if the user provided a classloader through the API, use that one
    if (classLoader != null) {
      ReflectUtil.log.trace("using provided classloader");
      return classLoader;
    }

    // 2) if the user provided a classloader through the environment, use that
    // one
    final Environment environment = Environment.getCurrent();
    if (environment != null) {
      classLoader = environment.getClassLoader();
      if (classLoader != null) {
        ReflectUtil.log.trace("using environment classloader");
        return classLoader;
      }
    }

    // 3) otherwise, use the current thread's context classloader
    ReflectUtil.log.trace("using context classloader");
    return Thread.currentThread().getContextClassLoader();
  }

  public static Class< ? > loadClass(ClassLoader classLoader, final String className) {
    try {
      classLoader = ReflectUtil.resolveClassLoader(classLoader);
      ReflectUtil.log.trace("loading class " + className);
      return classLoader.loadClass(className);
    } catch (final NoClassDefFoundError e) {
      throw new PvmException("couldn't define class " + className, e);
    } catch (final ClassNotFoundException e) {
      throw new PvmException("couldn't load class " + className, e);
    }
  }

  public static InputStream getResourceAsStream(ClassLoader classLoader,
      final String resource) {
    classLoader = ReflectUtil.resolveClassLoader(classLoader);
    ReflectUtil.log.trace("getting resource as stream " + resource);
    return classLoader.getResourceAsStream(resource);
  }

  public static Enumeration<URL> getResources(ClassLoader classLoader,
      final String resource) {
    classLoader = ReflectUtil.resolveClassLoader(classLoader);
    try {
      ReflectUtil.log.trace("getting resources " + resource);
      return classLoader.getResources(resource);
    } catch (final Exception e) {
      throw new PvmException("couldn't get resources " + resource, e);
    }
  }

  public static URL getResource(ClassLoader classLoader, final String resource) {
    classLoader = ReflectUtil.resolveClassLoader(classLoader);
    try {
      ReflectUtil.log.trace("getting resource " + resource);
      return classLoader.getResource(resource);
    } catch (final Exception e) {
      throw new PvmException("couldn't get resources " + resource, e);
    }
  }

  public static Object instantiate(ClassLoader classLoader, final String className) {
    Object newObject;
    try {
      classLoader = ReflectUtil.resolveClassLoader(classLoader);
      final Class< ? > clazz = ReflectUtil.loadClass(classLoader, className);
      ReflectUtil.log.trace("instantiating " + className);
      newObject = clazz.newInstance();
    } catch (final Exception e) {
      throw new PvmException("couldn't instantiate " + className, e);
    }
    return newObject;
  }

  public static Class< ? >[] loadClasses(ClassLoader classLoader,
      final List<String> constructorArgTypeNames) {
    if (constructorArgTypeNames == null) {
      return null;
    }
    final Class< ? >[] classes = new Class[constructorArgTypeNames.size()];
    for (int i = 0; i < constructorArgTypeNames.size(); i++) {
      classLoader = ReflectUtil.resolveClassLoader(classLoader);
      classes[i] = ReflectUtil.loadClass(classLoader, constructorArgTypeNames.get(i));
    }
    return classes;
  }

  public static <T> Constructor<T> getConstructor(final Class<T> clazz,
      final Class< ? >[] parameterTypes) {
    Constructor<T> constructor = null;
    try {
      constructor = clazz.getDeclaredConstructor(parameterTypes);

      if (ReflectUtil.log.isTraceEnabled()) {
        ReflectUtil.log.trace("found constructor " + clazz.getName() + "("
            + ArrayUtil.toString(parameterTypes) + ")");
      }

    } catch (final SecurityException e) {
      throw new PvmException("wasn't allowed to get constructor '"
          + clazz.getName() + "(" + ReflectUtil.getParameterTypesText(parameterTypes)
          + ")'", e);
    } catch (final NoSuchMethodException e) {
      throw new PvmException("couldn't find constructor '" + clazz.getName()
          + "(" + ReflectUtil.getParameterTypesText(parameterTypes) + ")'", e);
    }

    return constructor;
  }

  public static Field getField(final Class< ? > clazz, final String fieldName) {
    return ReflectUtil.getField(clazz, fieldName, clazz);
  }

  private static Field getField(final Class< ? > clazz, final String fieldName,
      final Class< ? > original) {
    Field field = null;

    try {
      field = clazz.getDeclaredField(fieldName);
      ReflectUtil.log.trace("found field " + fieldName + " in " + clazz.getName());
    } catch (final SecurityException e) {
      throw new PvmException("wasn't allowed to get field '" + clazz.getName()
          + "." + fieldName + "'", e);
    } catch (final NoSuchFieldException e) {
      if (clazz.getSuperclass() != null) {
        return ReflectUtil.getField(clazz.getSuperclass(), fieldName, original);
      } else {
        throw new PvmException("couldn't find field '" + original.getName()
            + "." + fieldName + "'", e);
      }
    }

    return field;
  }

  public static Method getMethod(final Class< ? > clazz, final String methodName,
      final Class< ? >[] parameterTypes) {
    return ReflectUtil.getMethod(clazz, methodName, parameterTypes, clazz);
  }

  private static Method getMethod(final Class< ? > clazz, final String methodName,
      final Class< ? >[] parameterTypes, final Class< ? > original) {
    Method method = null;

    try {
      method = clazz.getDeclaredMethod(methodName, parameterTypes);

      if (ReflectUtil.log.isTraceEnabled()) {
        ReflectUtil.log.trace("found method " + clazz.getName() + "." + methodName + "("
            + ArrayUtil.toString(parameterTypes) + ")");
      }

    } catch (final SecurityException e) {
      throw new PvmException("wasn't allowed to get method '" + clazz.getName()
          + "." + methodName + "(" + ReflectUtil.getParameterTypesText(parameterTypes)
          + ")'", e);
    } catch (final NoSuchMethodException e) {
      if (clazz.getSuperclass() != null) {
        return ReflectUtil.getMethod(clazz.getSuperclass(), methodName, parameterTypes,
            original);
      } else {
        throw new PvmException("couldn't find method '" + original.getName()
            + "." + methodName + "(" + ReflectUtil.getParameterTypesText(parameterTypes)
            + ")'", e);
      }
    }

    return method;
  }

  private static String getParameterTypesText(final Class< ? >[] parameterTypes) {
    if (parameterTypes == null) {
      return "";
    }
    final StringBuffer parametersTypeText = new StringBuffer();
    for (int i = 0; i < parameterTypes.length; i++) {
      final Class< ? > parameterType = parameterTypes[i];
      parametersTypeText.append(parameterType.getName());
      if (i != parameterTypes.length - 1) {
        parametersTypeText.append(", ");
      }
    }
    return parametersTypeText.toString();
  }

  public static <T> T newInstance(final Class<T> clazz) {
    return ReflectUtil.newInstance(clazz, null, null);
  }

  public static <T> T newInstance(final Constructor<T> constructor) {
    return ReflectUtil.newInstance(null, constructor, null);
  }

  public static <T> T newInstance(final Constructor<T> constructor, final Object[] args) {
    return ReflectUtil.newInstance(null, constructor, args);
  }

  private static <T> T newInstance(final Class<T> clazz, Constructor<T> constructor,
      final Object[] args) {
    if ((clazz == null) && (constructor == null)) {
      throw new IllegalArgumentException(
          "can't create new instance without clazz or constructor");
    }

    String className = null;
    try {
      ReflectUtil.log.trace("creating new instance for class '" + className
          + "' with args " + ArrayUtil.toString(args));
      if (constructor == null) {
        ReflectUtil.log.trace("getting default constructor");
        constructor = clazz.getConstructor((Class[]) null);
      }
      className = constructor.getDeclaringClass().getName();
      if (!constructor.isAccessible()) {
        ReflectUtil.log.trace("making constructor accessible");
        constructor.setAccessible(true);
      }
      return constructor.newInstance(args);

    } catch (final Exception t) {
      throw new PvmException("couldn't construct new '" + className
          + "' with args " + ArrayUtil.toString(args), t);
    }
  }

  public static Object get(final Field field, final Object object) {
    if (field == null) {
      throw new NullPointerException("field is null");
    }
    try {
      final Object value = field.get(object);
      ReflectUtil.log.trace("got value '" + value + "' from field '" + field.getName()
          + "'");
      return value;
    } catch (final Exception e) {
      throw new PvmException("couldn't get '" + field.getName() + "'", e);
    }
  }

  public static void set(final Field field, final Object object, final Object value) {
    if (field == null) {
      throw new NullPointerException("field is null");
    }
    try {
      ReflectUtil.log.trace("setting field '" + field.getName() + "' to value '" + value
          + "'");
      if (!field.isAccessible()) {
        ReflectUtil.log.trace("making field accessible");
        field.setAccessible(true);
      }
      field.set(object, value);
    } catch (final Exception e) {
      throw new PvmException("couldn't set '" + field.getName() + "' to '"
          + value + "'", e);
    }
  }

  public static Object invoke(final Method method, final Object target, final Object[] args) {
    if (method == null) {
      throw new PvmException("method is null");
    }
    try {
      ReflectUtil.log.trace("invoking '" + method.getName() + "' on '" + target + "' with "
          + ArrayUtil.toString(args));
      if (!method.isAccessible()) {
        ReflectUtil.log.trace("making method accessible");
        method.setAccessible(true);
      }
      return method.invoke(target, args);
    } catch (final InvocationTargetException e) {
      final Throwable targetException = e.getTargetException();
      throw new PvmException("couldn't invoke '" + method.getName() + "' with "
          + ArrayUtil.toString(args) + " on " + target + ": "
          + targetException.getMessage(), targetException);
    } catch (final Exception e) {
      throw new PvmException("couldn't invoke '" + method.getName() + "' with "
          + ArrayUtil.toString(args) + " on " + target + ": " + e.getMessage(),
          e);
    }
  }

  public static Method findMethod(final Class< ? > clazz, final String methodName,
      final List<ArgDescriptor> argDescriptors, final Object[] args) {
    ReflectUtil.log.trace("searching for method " + methodName + " in " + clazz.getName());
    final Method[] candidates = clazz.getDeclaredMethods();
    for (final Method candidate : candidates) {
      if ((candidate.getName().equals(methodName))
          && (ReflectUtil.isArgumentMatch(candidate.getParameterTypes(), argDescriptors,
              args))) {

        if (ReflectUtil.log.isTraceEnabled()) {
          ReflectUtil.log.trace("found matching method " + clazz.getName() + "."
              + methodName);
        }

        return candidate;
      }
    }
    if (clazz.getSuperclass() != null) {
      return ReflectUtil.findMethod(clazz.getSuperclass(), methodName, argDescriptors, args);
    }
    return null;
  }

  public static Constructor< ? > findConstructor(final Class< ? > clazz,
      final List<ArgDescriptor> argDescriptors, final Object[] args) {
    final Constructor< ? >[] constructors = clazz.getDeclaredConstructors();
    for (final Constructor< ? > constructor : constructors) {
      if (ReflectUtil.isArgumentMatch(constructor.getParameterTypes(), argDescriptors,
          args)) {
        return constructor;
      }
    }
    return null;
  }

  public static boolean isArgumentMatch(final Class< ? >[] parameterTypes,
      final List<ArgDescriptor> argDescriptors, final Object[] args) {
    int nbrOfArgs = 0;
    if (args != null) {
      nbrOfArgs = args.length;
    }

    int nbrOfParameterTypes = 0;
    if (parameterTypes != null) {
      nbrOfParameterTypes = parameterTypes.length;
    }

    if ((nbrOfArgs == 0) && (nbrOfParameterTypes == 0)) {
      return true;
    }

    if (nbrOfArgs != nbrOfParameterTypes) {
      return false;
    }

    for (int i = 0; i < parameterTypes.length; i++) {
      final Class< ? > parameterType = parameterTypes[i];
      final String argTypeName = argDescriptors != null ? argDescriptors.get(i).getTypeName() : null;
      if (argTypeName != null) {
        if (!argTypeName.equals(parameterType.getName())) {
          return false;
        }
      } else if ((args[i] != null)
          && (!parameterType.isAssignableFrom(args[i].getClass()))) {
        return false;
      }
    }
    return true;
  }

  public static String getSignature(final String methodName,
      final List<ArgDescriptor> argDescriptors, final Object[] args) {
    String signature = methodName + "(";
    if (args != null) {
      for (int i = 0; i < args.length; i++) {
        String argType = null;
        if (argDescriptors != null) {
          final ArgDescriptor argDescriptor = argDescriptors.get(i);
          if ((argDescriptor != null) && (argDescriptor.getTypeName() != null)) {
            argType = argDescriptor.getTypeName();
          }
        }
        if ((argType == null) && (args[i] != null)) {
          argType = args[i].getClass().getName();
        }
        signature += argType;
        if (i < (args.length - 1)) {
          signature += ", ";
        }
      }
    }
    signature += ")";
    return signature;
  }

  public static String getUnqualifiedClassName(final Class< ? > clazz) {
    if (clazz == null) {
      return null;
    }
    return ReflectUtil.getUnqualifiedClassName(clazz.getSimpleName());
  }

  public static String getUnqualifiedClassName(String className) {
    if (className == null) {
      return null;
    }
    final int dotIndex = className.lastIndexOf('.');
    if (dotIndex != -1) {
      className = className.substring(dotIndex + 1);
    }
    return className;
  }
}
