/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.ejb;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;

import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.env.EnvironmentFactory;
import org.ow2.orchestra.pvm.env.PvmEnvironmentFactory;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * Stateless session command executor.
 *
 * <h3>Configuration</h3>
 *
 * <p>
 * In order to provide commands an environment in which to run, the command
 * executor builds an environment factory from a configuration file loaded as a
 * classpath resource. The environment entry <code>ConfigurationResource</code>
 * specifies the name of the resource to access; the default is
 * <code>enterprise.environment.cfg.xml</code>.
 * </p>
 *
 * <h3>JNDI-bound environment factory</h3>
 *
 * <p>
 * To avoid parsing the configuration file multiple times, the command executor
 * attempts to bind the environment factory to the name specified in the
 * <code>EnvironmentFactoryName</code> environment entry, defaulting to
 * <code>java:orchestra/EnvironmentFactory</code>. If the binding fails, the
 * command executor will still work; however, each instance will build its own
 * environment factory from the configuration.
 * </p>
 *
 * @author Jim Rigsbee
 * @author Tom Baeyens
 * @author Alejandro Guizar
 */
public class CommandExecutorSLSB implements SessionBean {

  private SessionContext sessionContext;
  private EnvironmentFactory environmentFactory;

  private static final long serialVersionUID = 1L;

  private static final Log LOG = Log.getLog(CommandExecutorSLSB.class.getName());

  public <T> T execute(final Command<T> command) {
    final Environment environment = this.environmentFactory.openEnvironment();
    try {
      CommandExecutorSLSB.LOG.debug("executing command " + command);
      return command.execute(environment);
    } catch (final RuntimeException e) {
      throw e;
    } catch (final Exception e) {
      throw new EJBException("failed to execute command " + command, e);
    } finally {
      environment.close();
    }
  }

  public void setSessionContext(final SessionContext sessionContext) {
    this.sessionContext = sessionContext;
  }

  /**
   * Creates a command executor that will be used to execute the commands that
   * are passed in the execute method.
   */
  public void ejbCreate() throws CreateException {
    String envFactoryName = "java:orchestra/EnvironmentFactory";
    try {
      final Context initial = new InitialContext();

      // determine environment factory name
      try {
        envFactoryName = (String) initial
            .lookup("java:comp/env/EnvironmentFactoryName");
      } catch (final NameNotFoundException e) {
        CommandExecutorSLSB.LOG.debug("environment factory name not set, using default: "
            + envFactoryName);
      }

      try {
        // retrieve environment factory
        final Object namedObject = initial.lookup(envFactoryName);
        if (namedObject instanceof EnvironmentFactory) {
          CommandExecutorSLSB.LOG.debug("using environment factory at " + envFactoryName);
          this.environmentFactory = (EnvironmentFactory) namedObject;
        } else {
          /*
           * the bound object could be an instance of an environment factory
           * loaded in another (older?) deployment; because each deployment sets
           * up its own class loader, the existing instance could be perceived
           * as not being an environment factory
           */
          if (namedObject == null
              || CommandExecutorSLSB.isInstance(EnvironmentFactory.class.getName(), namedObject)) {
            CommandExecutorSLSB.LOG.debug("object bound to " + envFactoryName
                + " is a stale object factory, or null; unbinding it");
            initial.unbind(envFactoryName);

            this.environmentFactory = CommandExecutorSLSB.parseConfig(CommandExecutorSLSB.getConfigResource(initial));
            CommandExecutorSLSB.bind(initial, this.environmentFactory, envFactoryName);
          } else {
            CommandExecutorSLSB.LOG.debug("object bound to " + envFactoryName
                + " is not an environment factory, building one");
            this.environmentFactory = CommandExecutorSLSB.parseConfig(CommandExecutorSLSB.getConfigResource(initial));
            // no bind attempt
          }
        }
      } catch (final NameNotFoundException noEnv) {
        CommandExecutorSLSB.LOG.debug("environment factory not found at " + envFactoryName
            + ", building it");
        this.environmentFactory = CommandExecutorSLSB.parseConfig(CommandExecutorSLSB.getConfigResource(initial));
        CommandExecutorSLSB.bind(initial, this.environmentFactory, envFactoryName);
      }
    } catch (final NamingException e) {
      CommandExecutorSLSB.LOG.error("could not create command executor", e);
      throw new CreateException("jndi access failed");
    }
  }

  private static boolean isInstance(final String className, final Object object) {
    for (Class<?> cl = object.getClass(); cl != Object.class; cl = cl
        .getSuperclass()) {
      if (cl.getName().equals(className)) {
        return true;
      }
    }
    return false;
  }

  private static String getConfigResource(final Context context)
      throws NamingException {
    String resource = "enterprise.environment.cfg.xml";
    try {
      resource = (String) context.lookup("java:comp/env/ConfigurationResource");
    } catch (final NameNotFoundException e) {
      CommandExecutorSLSB.LOG.debug("configuration resource not set, using default: " + resource);
    }
    return resource;
  }

  private static EnvironmentFactory parseConfig(final String resource) {
    CommandExecutorSLSB.LOG.debug("parsing configuration from " + resource);
    return new PvmEnvironmentFactory(resource);
  }

  private static void bind(final Context context,
      final EnvironmentFactory environmentFactory, final String name) {
    try {
      context.bind(name, environmentFactory);
      CommandExecutorSLSB.LOG.info("bound " + environmentFactory + " to " + name);
    } catch (final NamingException e) {
      CommandExecutorSLSB.LOG.info("WARNING: environment factory binding failed", e);
    }
  }

  public void ejbRemove() {
    this.environmentFactory = null;
    this.sessionContext = null;
  }

  public void ejbActivate() {
  }

  public void ejbPassivate() {
  }
}
