/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.job.JobImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.model.ExecutionImpl;

/**
 * @author Tom Baeyens
 */
public class AcquireJobsCmd implements Command<Collection<Collection<JobImpl<?>>>> {

  private static final long serialVersionUID = 1L;

  private static final Log LOG = Log.getLog(AcquireJobsCmd.class.getName());
  private final DateFormat timeFormat = new SimpleDateFormat("HH:mm:ss,SSS");

  private final String lockOwnerName;
  private final int lockMillis;

  private final AbstractJobExecutor jobExecutor;
  private final boolean limitSameInstanceJobs;

  public AcquireJobsCmd(final AbstractJobExecutor jobExecutor, final String lockOwnerName, final int lockMillis, final boolean limitSameInstanceJobs) {
    this.jobExecutor = jobExecutor;
    this.lockOwnerName = lockOwnerName;
    this.lockMillis = lockMillis;
    this.limitSameInstanceJobs = limitSameInstanceJobs;
  }

  public Collection<Collection<JobImpl<?>>> execute(final Environment environment) {
    final Collection<Collection<JobImpl<?>>> result = new ArrayList<Collection<JobImpl<?>>>();

    final JobDbSession jobDbSession = environment.get(JobDbSession.class);
    AcquireJobsCmd.LOG.debug("start querying first acquirable jobImpl...");

    final int limit = this.jobExecutor.getAcquireJobSize();
    final Collection<JobImpl< ? >> jobs = jobDbSession.findFirstAcquirableJob(limit, this.limitSameInstanceJobs);

    if (jobs != null && !jobs.isEmpty()) {
      final Collection<ExecutionImpl> instances = new HashSet<ExecutionImpl>();
      for (final JobImpl<?> job : jobs) {
        if (instances.contains(job.getProcessInstance())) {
          continue;
        }
        instances.add(job.getProcessInstance());
        final Collection<JobImpl<?>> acquiredJobDbids = new ArrayList<JobImpl<?>>();
        final Collection<JobImpl< ? >> acquiredJobs = new ArrayList<JobImpl< ? >>();

        if (job.isExclusive()) {
          AcquireJobsCmd.LOG
              .trace("exclusive acquirable jobImpl found ("
                  + job
                  + "). querying for other exclusive jobs to lock them all in one tx...");
          final List<JobImpl< ? >> otherExclusiveJobs = jobDbSession
              .findExclusiveJobs(job.getProcessInstance());
          acquiredJobs.addAll(otherExclusiveJobs);
        } else {
          acquiredJobs.add(job);
        }

        for (final JobImpl< ? > acquiredJob : acquiredJobs) {
          final long lockExpirationTime = System.currentTimeMillis()
              + this.lockMillis;
          AcquireJobsCmd.LOG.trace("trying to obtain a lock for '" + acquiredJob + "' with exp "
              + this.timeFormat.format(new Date(lockExpirationTime)));
          acquiredJob.setLockExpirationTime(new Date(lockExpirationTime));
          acquiredJob.setLockOwner(this.lockOwnerName);
          acquiredJobDbids.add(acquiredJob);
        }
        result.add(acquiredJobDbids);
      }
    } else {
      AcquireJobsCmd.LOG.trace("no acquirable jobs in jobImpl table");
    }

    AcquireJobsCmd.LOG.debug("locking jobs " + result);

    return result;
  }

}
