/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Date;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.env.Transaction;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.internal.job.JobImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.svc.RetryInterceptor;

/**
 * @author Tom Baeyens
 */
public class JobExceptionHandler implements Command<Object> {

  private static final Log LOG = Log.getLog(JobExceptionHandler.class.getName());
  private static final long serialVersionUID = 1L;

  protected final long jobDbid;
  protected final JobImpl< ? > failedJob;
  protected final Throwable exception;

  public JobExceptionHandler(final long jobDbid, final Throwable exception) {
    this.jobDbid = jobDbid;
    this.exception = exception;
    this.failedJob = null;
  }

  public JobExceptionHandler(final JobImpl< ? > job, final Throwable exception) {
    this.jobDbid = job.getDbid();
    this.failedJob = job;
    this.exception = exception;
  }

  public Object execute(final Environment environment) {
    JobExceptionHandler.LOG.debug("handling jobImpl " + this.jobDbid + " exception: "
        + this.exception.getMessage());

    JobImpl<?> job = this.failedJob;
    if (job == null) {
      // load the jobImpl from the db
      final JobDbSession session = environment.get(JobDbSession.class);
      if (session == null) {
        throw new PvmException("no job-session configured to handle job");
      }
      job = session.get(JobImpl.class, this.jobDbid);
    }
    // serialize the stack trace
    final StringWriter sw = new StringWriter();
    this.exception.printStackTrace(new PrintWriter(sw));
    if (job != null) {
      // decrement the number of retries
      final int decrementedRetries = job.getRetries() - 1;
      JobExceptionHandler.LOG
          .debug("decrementing retries to " + decrementedRetries + " for "
              + job);
      job.setRetries(decrementedRetries);
      job.setException(sw.toString());
      JobExceptionHandler.LOG.debug("unlocking " + job);
      job.setLockOwner(null);
      job.setLockExpirationTime(null);

      final JobExecutor jobExecutor = environment.get(JobExecutor.class);
      if (decrementedRetries == 0) {
        JobExceptionHandler.LOG.error("All execution of job " + job + " failed."
            + "Last exception was " + job.getException());
        final DeadJobHandler deadJobHandler = jobExecutor.getDeadJobHandler();
        if (deadJobHandler != null) {
          deadJobHandler.handleDeadJob(job);
        }
      } else {
        // reschedule job for retry
        long delay = 50;
        int delayFactor = 2;
        if (jobExecutor instanceof AbstractJobExecutor) {
          final CommandService commandService = ((AbstractJobExecutor)jobExecutor).getCommandExecutor();
          if (commandService instanceof RetryInterceptor) {
            delay = ((RetryInterceptor) commandService).getDelay();
            delayFactor = ((RetryInterceptor) commandService).getDelayFactor();
          }
        }
        final long attemptFactor = job.getDelay() * (1 + RetryInterceptor.getRandom().nextInt(delayFactor - 1));
        job.setDelay(attemptFactor);
        job.setDueDate(new Date(System.currentTimeMillis() + delay * attemptFactor));
        // notify the job executor after the transaction is completed
        final Transaction transaction = environment.get(Transaction.class);
        if ((transaction != null) && (jobExecutor != null)) {
          JobExceptionHandler.LOG.trace("registering job executor notifier with " + transaction);
          transaction.registerSynchronization(new JobAddedNotification(
              jobExecutor));
        }
      }
    }
    return null;
  }
}
