/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.env.Transaction;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.model.ExecutionImpl.Propagation;
import org.ow2.orchestra.pvm.internal.model.op.MoveToChildNode;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.listener.EventListener;

/**
 * @author Tom Baeyens
 * @author Guillaume Porcher
 */
public class ExceptionHandlerImpl implements Serializable {

  private static final long serialVersionUID = 1L;
  static final Log LOG = Log.getLog(ExceptionHandlerImpl.class.getName());

  protected long dbid;
  protected int dbversion;
  protected String exceptionClassName;
  protected boolean isTransactional = false;
  protected boolean isRethrowMasked = false;
  protected List<ObjectReference<EventListener>> eventListenerReferences;
  protected String transitionName; // mutually exclusive with nodeName
  protected String nodeName; // mutually exclusive with transitionName

  // construction methods /////////////////////////////////////////////////////

  public ObjectReference<EventListener> createEventListenerReference(
      final EventListener eventListener) {
    final ObjectReference<EventListener> eventListenerReference = this.createEventListenerReference();
    eventListenerReference.set(eventListener);
    return eventListenerReference;
  }

  public ObjectReference<EventListener> createEventListenerReference(
      final Descriptor descriptor) {
    final ObjectReference<EventListener> eventListenerReference = this.createEventListenerReference();
    eventListenerReference.setDescriptor(descriptor);
    return eventListenerReference;
  }

  public ObjectReference<EventListener> createActivityReference(
      final String expression) {
    final ObjectReference<EventListener> eventListenerReference = this.createEventListenerReference();
    eventListenerReference.setExpression(expression);
    return eventListenerReference;
  }

  public ObjectReference<EventListener> createEventListenerReference() {
    if (this.eventListenerReferences == null) {
      this.eventListenerReferences = new ArrayList<ObjectReference<EventListener>>();
    }
    final ObjectReference<EventListener> actionObjectReference = new ObjectReference<EventListener>();
    this.eventListenerReferences.add(actionObjectReference);
    return actionObjectReference;
  }

  public List<EventListener> getEventListeners() {
    if (this.eventListenerReferences == null) {
      return null;
    }
    final List<EventListener> eventListeners = new ArrayList<EventListener>(
        this.eventListenerReferences.size());
    for (final ObjectReference<EventListener> eventListenerReference : this.eventListenerReferences) {
      final EventListener eventListener = eventListenerReference.get();
      eventListeners.add(eventListener);
    }
    return eventListeners;
  }

  // runtime behaviour methods ////////////////////////////////////////////////

  public boolean matches(final Exception exception) {
    return this.matches(exception.getClass());
  }

  public boolean matches(final Class< ? > exceptionClass) {
    if (exceptionClass == null) {
      return false;
    }
    final Class< ? > exceptionClassClass = ReflectUtil.loadClass(Thread.currentThread().getContextClassLoader(),
      this.exceptionClassName);
    return exceptionClassClass.isAssignableFrom(exceptionClass);
  }

  /**
   * Handles the exception. This method will execute compensating code
   * in the same TX or in another TX (depending on the value of {@link #isTransactional} field).
   * The compensating code is not in this method,
   * but in {@link #executeHandler(ExecutionImpl, Exception, ProcessElementImpl)} method.
   * @param execution the execution which produced the exception
   * @param exception the exception to handle
   * @param processElementImpl the process element the exception handler was triggered from.
   */
  public void handle(final ExecutionImpl execution, final Exception exception, final ProcessElementImpl processElementImpl) {
    if (this.isTransactional) {
      final Environment environment = Environment.getCurrent();
      final Transaction transaction = environment != null ? environment.get(Transaction.class) : null;
      if (transaction != null) {
        ExceptionHandlerImpl.LOG.trace("registering exception handler to " + transaction);
        final CommandService commandService = environment.get(CommandService.class);
        if (commandService == null) {
          throw new PvmException(
              "environment doesn't have a command service for registering transactional exception handler",
              exception);
        }
        final ExceptionHandlerSynchronization exceptionHandlerSynchronization = new ExceptionHandlerSynchronization(
            this, execution, exception, processElementImpl, commandService);
        // registration of the synchronization is delegated to the
        // AfterTxCompletionListener
        // to avoid a dependency on class Synchronization
        exceptionHandlerSynchronization.register(transaction);
        ExceptionHandlerImpl.LOG.trace("registering exception handler to " + transaction);
        throw new PvmException(
            "transaction exception handler registered handler after transaction completed.  "
            + "make sure this transaction is rolled back",
            exception);
      } else {
        throw new PvmException(
            "no transaction present in the environment for transactional exception handler",
            exception);
      }
    } else {
      this.executeHandler(execution, exception, processElementImpl);
    }
  }

  /**
   * Compensating code for the exception.
   * This method is called from {@link #handle(ExecutionImpl, Exception, ProcessElementImpl)}
   * or from {@link ExceptionHandlerSynchronization} and should not be called directly.
   *
   * Extensions should overwrite this method.
   *
   * @param execution the execution which produced the exception
   * @param exception the exception to handle
   * @param processElementImpl the process element the exception handler was triggered from.
   */
  protected void executeHandler(final ExecutionImpl execution, final Exception exception,
      final ProcessElementImpl processElementImpl) {
    if (this.eventListenerReferences != null) {
      for (final ObjectReference<EventListener> eventListenerReference : this.eventListenerReferences) {

        final EventListener eventListener = eventListenerReference.get();

        ExceptionHandlerImpl.LOG.trace("executing " + eventListener + " for " + this);
        try {
          eventListener.notify(execution);
        } catch (final RuntimeException e) {
          throw e;
        } catch (final Exception e) {
          throw new PvmException("couldn't execute " + eventListener, e);
        }
      }
    }

    if (this.transitionName != null || this.nodeName != null) {
      // perform atomic operation is only offering a new Operation.
      // execution of operation is not done when the method is returning.
      // as a result, we need to explicitly stop execution propagation
      // to be sure exception node has been executed.
      // Normal execution has to be resumed (if wanted) manually.
      execution.setPropagation(Propagation.EXPLICIT);
    }
    if (this.transitionName != null) {
      // a transition can be taken if and only if the exception handler is set on a node.
      NodeImpl node = null;
      if (processElementImpl != null && processElementImpl instanceof NodeImpl) {
        node = (NodeImpl) processElementImpl;
        final TransitionImpl transition = node.findOutgoingTransition(this.transitionName);
        if (transition != null) {
          ExceptionHandlerImpl.LOG.trace(this.toString() + " takes transition " + this.transitionName);
          execution.take(transition);
        } else {
          ExceptionHandlerImpl.LOG.info("WARNING: " + this.toString() + " couldn't find transition "
              + this.transitionName + " on " + node);
        }
      } else {
        ExceptionHandlerImpl.LOG.info("WARNING: " + this.toString()
            + " couldn't find current node to take transition "
            + this.transitionName);
      }
    } else if (this.nodeName != null) {
      // execute child node of process element
      NodeImpl childNode = null;
      if (processElementImpl != null) {
        if (processElementImpl instanceof CompositeElementImpl) {
          childNode = ((CompositeElementImpl) processElementImpl).getNode(this.nodeName);
        }
      }
      if (childNode != null) {
        ExceptionHandlerImpl.LOG.trace(this.toString() + " executes node " + this.nodeName);
        execution.performAtomicOperationSync(new MoveToChildNode(childNode));
      } else {
        ExceptionHandlerImpl.LOG.info("WARNING: " + this.toString() + " couldn't find child node "
            + this.nodeName);
      }
    }
  }

  public static void rethrow(final Exception exception, final String prefixMessage) {
    ExceptionHandlerImpl.LOG.trace("rethrowing " + exception);
    if (exception instanceof RuntimeException) {
      throw (RuntimeException) exception;
    } else {
      throw new PvmException(prefixMessage + ": " + exception.getMessage(),
          exception);
    }
  }

  @Override
  public String toString() {
    return this.exceptionClassName != null ?
        "exception-handler(" + this.exceptionClassName + ")" : "exception-handler";
  }

  // getters and setters //////////////////////////////////////////////////////

  public long getDbid() {
    return this.dbid;
  }

  public String getExceptionClassName() {
    return this.exceptionClassName;
  }

  public void setExceptionClassName(final String exceptionClassName) {
    this.exceptionClassName = exceptionClassName;
  }

  public boolean isTransactional() {
    return this.isTransactional;
  }

  public void setTransactional(final boolean isTransactional) {
    this.isTransactional = isTransactional;
  }

  public String getTransitionName() {
    return this.transitionName;
  }

  public void setTransitionName(final String transitionName) {
    this.transitionName = transitionName;
  }

  public String getNodeName() {
    return this.nodeName;
  }

  public void setNodeName(final String nodeName) {
    this.nodeName = nodeName;
  }

  public boolean isRethrowMasked() {
    return this.isRethrowMasked;
  }

  public void setRethrowMasked(final boolean isRethrowMasked) {
    this.isRethrowMasked = isRethrowMasked;
  }

  public List<ObjectReference<EventListener>> getEventListenerReferences() {
    return this.eventListenerReferences;
  }

  public void setEventListenerReferences(
      final List<ObjectReference<EventListener>> activityReferences) {
    this.eventListenerReferences = activityReferences;
  }
}
