/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.activity.Activity;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.model.Node;
import org.ow2.orchestra.pvm.model.Transition;

/**
 * @author Tom Baeyens
 */
public class NodeImpl extends CompositeElementImpl implements Node {

  private static final long serialVersionUID = 1L;

  protected ObjectReference<Activity> behaviourReference;
  protected List<TransitionImpl> outgoingTransitions;
  protected List<TransitionImpl> incomingTransitions;
  protected TransitionImpl defaultTransition;
  protected NodeImpl parentNode;

  protected boolean isLocalScope;
  protected boolean isExecutionAsync;
  protected boolean isSignalAsync;
  protected boolean isLeaveAsync;
  protected boolean isPreviousNeeded;

  protected transient Map<String, TransitionImpl> outgoingTransitionsMap;

  /**
   * Use {@link ProcessDefinitionImpl#createNode()} or
   * {@link NodeImpl#createNode()} instead.
   */
  public NodeImpl() {
    super();
  }

  // specialized node containment methods /////////////////////////////////////

  @Override
  public NodeImpl addNode(final NodeImpl node) {
    node.setParentNode(this);
    super.addNode(node);
    return node;
  }

  @Override
  public Node findNode(final String nodeName) {
    if (nodeName == null) {
      if (this.name == null) {
        return this;
      }
    } else if (nodeName.equals(this.name)) {
      return this;
    }
    return super.findNode(nodeName);
  }

  // leaving transitions //////////////////////////////////////////////////////

  /**
   * creates a transition from this node to the given destination node. Also the
   * transition pointers to source and destination node will be set
   * appropriatly.
   *
   * @throws NullPointerException
   *           if destination is null.
   */
  public Transition createOutgoingTransition(final NodeImpl destination) {
    return this.createOutgoingTransition(destination, null);
  }

  /**
   * creates a transition with the given name from this node to the given
   * destination node. Also the transition pointers to source and destination
   * node will be set appropriatly.
   *
   * @param transitionName
   *          may be null.
   * @throws NullPointerException
   *           if destination is null.
   */
  public TransitionImpl createOutgoingTransition(final NodeImpl destination,
      final String transitionName) {
    final TransitionImpl transition = this.createOutgoingTransition(transitionName);
    if (destination != null) {
      destination.addIncomingTransition(transition);
    }
    return transition;
  }

  public TransitionImpl createOutgoingTransition(final String transitionName) {
    // create a new transition
    final TransitionImpl transition = new TransitionImpl();
    transition.setName(transitionName);

    // wire it between the source and destination
    this.addOutgoingTransition(transition);

    // if there is no default transition yet
    if (this.defaultTransition == null) {
      // make this the default outgoing transition
      this.defaultTransition = transition;
    }

    return transition;
  }

  /**
   * adds the given transition as a leaving transition to this node. Also the
   * source of the transition is set to this node. Adding a transition that is
   * already contained in the leaving transitions has no effect.
   *
   * @return the added transition.
   * @throws NullPointerException
   *           if transition is null.
   */
  public Transition addOutgoingTransition(final TransitionImpl transition) {
    transition.setSource(this);

    if (this.outgoingTransitions == null) {
      this.outgoingTransitions = new ArrayList<TransitionImpl>();
    }
    if (!this.outgoingTransitions.contains(transition)) {
      this.outgoingTransitions.add(transition);
    }
    this.outgoingTransitionsMap = null;
    return transition;
  }

  /**
   * removes the given transition from the leaving transitions. Also the
   * transition's source will be nulled. This method will do nothing if the
   * transition is null or if the given transition is not in the list of this
   * node's leaving transitions. In case this is the transition that was in the
   * outgoingTransitionsMap and another transition exists with the same name,
   * that transition (the first) will be put in the outgoingTransitionsMap as a
   * replacement for the removed transition. If the transition is actually
   * removed from the list of leaving transitions, the transition's source will
   * be nulled.
   */
  public boolean removeOutgoingTransition(final TransitionImpl transition) {
    if ((transition != null) && (this.outgoingTransitions != null)) {
      final boolean isRemoved = this.outgoingTransitions.remove(transition);
      if (isRemoved) {
        transition.setSource(null);
        if (this.outgoingTransitions.isEmpty()) {
          this.outgoingTransitions = null;
        }
        this.outgoingTransitionsMap = null;
      }
      return isRemoved;
    }
    return false;
  }

  /**
   * the first leaving transition with the given name or null of no such leaving
   * transition exists.
   */
  public TransitionImpl getOutgoingTransition(final String transitionName) {
    return this.getOutgoingTransitionsMap() != null ? this.outgoingTransitionsMap.get(transitionName) : null;
  }

  /**
   * searches for the given transitionName in this node and then up the parent
   * chain. Returns null if no such transition is found.
   */
  public TransitionImpl findOutgoingTransition(final String transitionName) {
    final TransitionImpl transition = this.getOutgoingTransition(transitionName);
    if (transition != null) {
      return transition;
    }
    if (this.parentNode != null) {
      return this.parentNode.findOutgoingTransition(transitionName);
    }
    return null;
  }

  /**
   * searches for the default transition in this node and then up the parent
   * chain. Returns null if no such transition is found.
   */
  public TransitionImpl findDefaultTransition() {
    if (this.defaultTransition != null) {
      return this.defaultTransition;
    }
    if (this.parentNode != null) {
      return this.parentNode.findDefaultTransition();
    }
    return null;
  }

  /**
   * the list of leaving transitions. Beware: the actual member is returned. No
   * copy is made.
   */
  public List<Transition> getOutgoingTransitions() {
    return (List) this.outgoingTransitions;
  }

  /** indicates if a leaving transition with the given transitionName exists. */
  public boolean hasOutgoingTransition(final String transitionName) {
    return this.getOutgoingTransition(transitionName) != null;
  }

  /** indicates if this node has leaving transitions */
  public boolean hasOutgoingTransitions() {
    return (this.outgoingTransitions != null) && (!this.outgoingTransitions.isEmpty());
  }

  /**
   * sets the outgoingTransitions to the given list of outgoingTransitions. A
   * copy of the collection is made. Also the outgoingTransitionsMap will be
   * updated and the source of all the transitions in the given list will be set
   * to this node. In case there was a leaving transitions list present, these
   * transition's source will be nulled.
   */
  public void setOutgoingTransitions(final List<TransitionImpl> outgoingTransitions) {
    if (this.outgoingTransitions != null) {
      for (final TransitionImpl removedTransition : this.outgoingTransitions) {
        removedTransition.setSource(null);
      }
    }
    if (outgoingTransitions != null) {
      this.outgoingTransitions = new ArrayList<TransitionImpl>(
          outgoingTransitions);
      for (final TransitionImpl addedTransition : outgoingTransitions) {
        addedTransition.setSource(this);
      }
    } else {
      this.outgoingTransitions = null;
    }
    this.outgoingTransitionsMap = null;
  }

  /**
   * the leaving transitions, keyed by transition name. If a transition with the
   * same name occurs mutltiple times, the first one is returned. Leaving
   * transitions with a null value for their name are not included in the map.
   * Beware: the actual member is returned. No copy is made.
   */
  public Map<String, Transition> getOutgoingTransitionsMap() {
    if (this.outgoingTransitionsMap == null) {
      this.outgoingTransitionsMap = NodeImpl.getTransitionsMap(this.outgoingTransitions);
    }
    return (Map) this.outgoingTransitionsMap;
  }

  // arriving transitions /////////////////////////////////////////////////////

  /**
   * adds the given transition as an arriving transition to this node. Also the
   * source of the transition is set to this node.
   *
   * @return the added transition.
   * @throws NullPointerException
   *           if transition is null.
   */
  public Transition addIncomingTransition(final TransitionImpl transition) {
    transition.setDestination(this);
    if (this.incomingTransitions == null) {
      this.incomingTransitions = new ArrayList<TransitionImpl>();
    }
    this.incomingTransitions.add(transition);
    return transition;
  }

  /**
   * removes the given transition if it is contained in the arriving transitions
   * of this node. If this transition was actually removed, its destination
   * pointer is nulled.
   *
   * @return true if a transition was removed.
   */
  public boolean removeIncomingTransition(final TransitionImpl transition) {
    if ((transition != null) && (this.incomingTransitions != null)
        && (this.incomingTransitions.remove(transition))) {
      transition.setDestination(null);
      if (this.incomingTransitions.isEmpty()) {
        this.incomingTransitions = null;
      }
      return true;
    }
    return false;
  }

  /**
   * the list of arriving transitions. Beware: the actual member is returned. No
   * copy is made.
   */
  public List<Transition> getIncomingTransitions() {
    return (List) this.incomingTransitions;
  }

  /** indicates if this node has arriving transitions */
  public boolean hasIncomingTransitions() {
    return (this.incomingTransitions != null) && (!this.incomingTransitions.isEmpty());
  }

  /**
   * sets the incomingTransitions to the given list of incomingTransitions. A
   * copy of the collection is made. Also the destination of all the transitions
   * in the given list will be set to this node. In case there was an arriving
   * transitions list present, these transition's destination will be nulled.
   */
  public void setIncomingTransitions(final List<TransitionImpl> incomingTransitions) {
    if (this.incomingTransitions != null) {
      for (final TransitionImpl removedTransition : this.incomingTransitions) {
        removedTransition.setDestination(null);
      }
    }
    if (incomingTransitions != null) {
      this.incomingTransitions = new ArrayList<TransitionImpl>(
          incomingTransitions);
      for (final TransitionImpl addedTransition : incomingTransitions) {
        addedTransition.setDestination(this);
      }
    } else {
      this.incomingTransitions = null;
    }
  }

  // behaviour ////////////////////////////////////////////////////////////////

  /**
   * sets the given activity as the behaviour for this node. An object reference
   * for the given activity is created.
   */
  public void setBehaviour(final Activity activity) {
    this.behaviourReference = new ObjectReference<Activity>(activity);
  }

  /**
   * sets the activity that can be created from the given descriptor as the
   * behaviour for this node. It is assumed that the descriptor will create an
   * {@link Activity} An object reference for the given descriptor is created.
   */
  public void setBehaviour(final Descriptor descriptor) {
    this.behaviourReference = new ObjectReference<Activity>(descriptor);
  }

  /**
   * sets the expression behaviour for this node. The evaluation of the
   * expression will replace the
   * {@link Activity#execute(org.ow2.orchestra.pvm.Execution) Activity's execute
   * method}. An object reference for the given descriptor is created.
   */
  public void setBehaviour(final String expression) {
    this.behaviourReference = new ObjectReference<Activity>(expression);
  }

  public ObjectReference<Activity> getBehaviourReference() {
    return this.behaviourReference;
  }

  public void setBehaviourReference(final ObjectReference<Activity> behaviourReference) {
    this.behaviourReference = behaviourReference;
  }

  public Activity getBehaviour() {
    final Activity behaviour = this.behaviourReference != null ? this.behaviourReference.get() : null;
    if (behaviour == null) {
      throw new PvmException("no behaviour on " + this);
    }
    return behaviour;
  }

  // various helper methods ///////////////////////////////////////////////////

  private static Map<String, TransitionImpl> getTransitionsMap(
      final List<TransitionImpl> transitions) {
    Map<String, TransitionImpl> map = null;
    if (transitions != null) {
      map = new HashMap<String, TransitionImpl>();
      for (final TransitionImpl transition : transitions) {
        if (!map.containsKey(transition.getName())) {
          map.put(transition.getName(), transition);
        }
      }
    }
    return map;
  }

  static Map<String, NodeImpl> getNodesMap(final List<NodeImpl> nodes) {
    Map<String, NodeImpl> map = null;
    if (nodes != null) {
      map = new HashMap<String, NodeImpl>();
      for (final NodeImpl node : nodes) {
        if (node.getName() != null) {
          if (!map.containsKey(node.getName())) {
            map.put(node.getName(), node);
          }
        }
      }
    }
    return map;
  }

  @Override
  public String toString() {
    if (this.name != null) {
      return "node(" + this.name + ")";
    }
    if (this.dbid != 0) {
      return "node(" + this.dbid + ")";
    }
    return "node(" + System.identityHashCode(this) + ")";
  }

  /**
   * collects the full stack of parent in a list. This node is the first element
   * in the chain. The process definition will be the last element. the chain
   * will never be null.
   */
  public List<ObservableElementImpl> getParentChain() {
    final List<ObservableElementImpl> chain = new ArrayList<ObservableElementImpl>();
    ObservableElementImpl processElement = this;
    while (processElement != null) {
      chain.add(processElement);
      processElement = processElement.getParent();
    }
    return chain;
  }

  // getters and setters //////////////////////////////////////////////////////

  @Override
  public ObservableElementImpl getParent() {
    return this.parentNode != null ? this.parentNode : this.processDefinition;
  }

  @Override
  public String getName() {
    return this.name;
  }

  @Override
  public void setName(final String name) {
    // if there is no processDefinition associated with this node
    if (this.processDefinition == null) {
      // it s just a setter
      this.name = name;

    } else { // otherwise
      // make sure the processDefinition's activitiesMap remains up to date
      if (this.name != null) {
        this.processDefinition.removeNode(this);
      }
      this.name = name;
      if (name != null) {
        this.processDefinition.addNode(this);
      }
    }
  }

  public TransitionImpl getDefaultTransition() {
    return this.defaultTransition;
  }

  public void setDefaultTransition(final TransitionImpl defaultTransition) {
    this.defaultTransition = defaultTransition;
  }

  public NodeImpl getParentNode() {
    return this.parentNode;
  }

  public void setParentNode(final NodeImpl parentNode) {
    this.parentNode = parentNode;
  }

  public boolean isExecutionAsync() {
    return this.isExecutionAsync;
  }

  public boolean isSignalAsync() {
    return this.isSignalAsync;
  }

  public void setSignalAsync(final boolean isSignalAsync) {
    this.isSignalAsync = isSignalAsync;
  }

  public void setExecutionAsync(final boolean isExecutionAsync) {
    this.isExecutionAsync = isExecutionAsync;
  }

  public boolean isLeaveAsync() {
    return this.isLeaveAsync;
  }

  public void setLeaveAsync(final boolean isLeaveAsync) {
    this.isLeaveAsync = isLeaveAsync;
  }

  public boolean isPreviousNeeded() {
    return this.isPreviousNeeded;
  }

  public void setPreviousNeeded(final boolean isPreviousNeeded) {
    this.isPreviousNeeded = isPreviousNeeded;
  }

  public boolean isLocalScope() {
    return this.isLocalScope;
  }

  public void setLocalScope(final boolean isLocalScope) {
    this.isLocalScope = isLocalScope;
  }
}
