/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.ow2.orchestra.pvm.Execution;
import org.ow2.orchestra.pvm.client.ClientProcessDefinition;
import org.ow2.orchestra.pvm.client.ClientProcessInstance;
import org.ow2.orchestra.pvm.internal.lob.Blob;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.IoUtil;
import org.ow2.orchestra.pvm.model.OpenProcessDefinition;

/**
 * @author Tom Baeyens
 */
public class ProcessDefinitionImpl extends CompositeElementImpl implements
    OpenProcessDefinition, ClientProcessDefinition {

  private static final long serialVersionUID = 1L;
  private static final Log LOG = Log.getLog(ProcessDefinitionImpl.class.getName());

  public static final int UNASSIGNED_VERSION = -1;

  /**
   * user provided short reference for the process definition that has the same
   * scope as the name. Multiple versions of the same process can have the same
   * key. The key is used to build the globally unique execution ids.
   */
  protected String key;

  /**
   * the unique id (e.g. combination of name and versionnumber) for this process
   * definition.
   */
  protected String id;

  /** the version number of the process definitions with the same name. */
  protected int version = ProcessDefinitionImpl.UNASSIGNED_VERSION;

  /** optional package name similar to the Java package name. */
  protected String packageName;

  /** time this process was deployed */
  protected Date deploymentTime;

  /** the node which is executed when the process starts */
  protected NodeImpl initial;

  /**
   * the attachments TODO move the attachments to a separate DeploymentUnit type
   */
  protected Map<String, Blob> attachments;

  public ProcessDefinitionImpl() {
    this.processDefinition = this;
  }

  // execution factory method /////////////////////////////////////////////////

  public ClientProcessInstance createProcessInstance() {
    return this.createProcessInstance(null, null);
  }

  public ClientProcessInstance createProcessInstance(final String key) {
    return this.createProcessInstance(key, null);
  }

  public ClientProcessInstance createProcessInstance(final String key,
      final Execution superProcessExecution) {
    final ExecutionImpl processInstance = this.newProcessInstance();
    ProcessDefinitionImpl.LOG.debug("creating new execution for process '" + this.name + "'");
    if (superProcessExecution != null) {
      // establish the bidirectional relation between super process activity
      // instance
      // and sub process instance
      final ExecutionImpl superProcessExecutionImpl = (ExecutionImpl) superProcessExecution;
      processInstance.setSuperProcessExecution(superProcessExecutionImpl);
      superProcessExecutionImpl.setSubProcessInstance(processInstance);
    }
    processInstance.initializeProcessInstance(this, key);
    return processInstance;
  }

  public ClientProcessInstance beginProcessInstance() {
    final ClientProcessInstance processInstance = this.createProcessInstance(null, null);
    processInstance.begin();
    return processInstance;
  }

  public ClientProcessInstance beginProcessInstance(final String key) {
    final ClientProcessInstance processInstance = this.createProcessInstance(key, null);
    processInstance.begin();
    return processInstance;
  }

  protected ExecutionImpl newProcessInstance() {
    return new ExecutionImpl();
  }

  // attachments //////////////////////////////////////////////////////////////

  public void addAttachment(final String name, final InputStream inputStream) {
    final byte[] bytes = IoUtil.readBytes(inputStream);
    this.addAttachment(name, bytes);
  }

  public void addAttachment(final String name, final byte[] bytes) {
    final Blob blob = new Blob(bytes);
    if (this.attachments == null) {
      this.attachments = new HashMap<String, Blob>();
    }
    this.attachments.put(name, blob);
  }

  public byte[] getAttachmentBytes(final String name) {
    if (this.attachments == null) {
      return null;
    }
    final Blob blob = this.attachments.get(name);
    if (blob == null) {
      return null;
    }
    return blob.getBytes();
  }

  public InputStream getAttachmentInputStream(final String name) {
    final byte[] bytes = this.getAttachmentBytes(name);
    if (bytes == null) {
      return null;
    }
    return new ByteArrayInputStream(bytes);
  }

  // basic methods ////////////////////////////////////////////////////////////

  @Override
  public String toString() {
    return this.name != null ? "processDefinition(" + this.name + ")" : "processDefinition";
  }

  // getters and setters //////////////////////////////////////////////////////

  public NodeImpl getInitial() {
    return this.initial;
  }

  public void setInitial(final NodeImpl initial) {
    this.initial = initial;
  }

  public int getVersion() {
    return this.version;
  }

  public void setVersion(final int version) {
    this.version = version;
  }

  public Date getDeploymentTime() {
    return this.deploymentTime;
  }

  public void setDeploymentTime(final Date deploymentTime) {
    this.deploymentTime = deploymentTime;
  }

  public String getPackageName() {
    return this.packageName;
  }

  public void setPackageName(final String packageName) {
    this.packageName = packageName;
  }

  public String getKey() {
    return this.key;
  }

  public void setKey(final String key) {
    this.key = key;
  }

  public String getId() {
    return this.id;
  }

  public void setId(final String id) {
    this.id = id;
  }
}
