/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.cal;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

/**
 * a calendar that knows about business hours.
 */
public class BusinessCalendar implements Serializable {

  private static final long serialVersionUID = 1L;
  private static BusinessCalendar instance = null;

  protected long oid = -1;
  protected int version = 0;
  protected TimeZone timeZone = TimeZone.getDefault();
  /**
   * array that contains the weekdays in the index as specified by
   * {@link Calendar#SUNDAY} (=1), {@link Calendar#MONDAY} (=2),...
   * {@link Calendar#SATURDAY} (=7).
   */
  protected Day[] days = null;
  protected Holiday[] holidays = null;

  protected long secondInMillis = 1000;
  protected long minuteInMillis = 60000;
  protected long hourInMillis = 3600000;
  protected long dayInMillis = 24 * this.hourInMillis;
  protected long weekInMillis = 7 * this.dayInMillis;
  protected long monthInMillis = 30 * this.dayInMillis;
  protected long yearInMillis = 365 * this.dayInMillis;

  protected long businessDayInMillis = 8 * this.hourInMillis;
  protected long businessWeekInMillis = 40 * this.hourInMillis;
  protected long businessMonthInMillis = 21 * this.dayInMillis;
  protected long businessYearInMillis = 220 * this.dayInMillis;

  /** constructor for persistence and creating an empty business calendar */
  public BusinessCalendar() {
  }

  public static synchronized BusinessCalendar getInstance() {
    if (BusinessCalendar.instance == null) {
      BusinessCalendar.instance = new BusinessCalendar();
    }
    return BusinessCalendar.instance;
  }

  public Date add(Date date, final Duration duration) {
    Date end = null;
    if (duration.isBusinessTime()) {
      DayPart dayPart = this.findDayPart(date);
      final boolean isInbusinessHours = dayPart != null;
      if (!isInbusinessHours) {
        final Object[] result = new Object[2];
        this.findDay(date).findNextDayPartStart(0, date, result);
        date = (Date) result[0];
        dayPart = (DayPart) result[1];
      }
      final long millis = this.convertToMillis(duration);
      end = dayPart.add(date, millis, duration.isBusinessTime());
    } else {
      final long millis = this.convertToMillis(duration);
      end = new Date(date.getTime() + millis);
    }
    return end;
  }

  public long convertToMillis(final Duration duration) {
    long millis = duration.getMillis();
    millis += duration.getSeconds() * this.secondInMillis;
    millis += duration.getMinutes() * this.minuteInMillis;
    millis += duration.getHours() * this.hourInMillis;
    if (duration.isBusinessTime()) {
      millis += duration.getDays() * this.businessDayInMillis;
      millis += duration.getWeeks() * this.businessWeekInMillis;
      millis += duration.getMonths() * this.businessMonthInMillis;
      millis += duration.getYears() * this.businessYearInMillis;
    } else {
      millis += duration.getDays() * this.dayInMillis;
      millis += duration.getWeeks() * this.weekInMillis;
      millis += duration.getMonths() * this.monthInMillis;
      millis += duration.getYears() * this.yearInMillis;
    }
    return millis;
  }

  public boolean isInBusinessHours(final Date date) {
    return this.findDayPart(date) != null;
  }

  public boolean isHoliday(final Date date) {
    if (this.holidays != null) {
      for (final Holiday holiday : this.holidays) {
        if (holiday.includes(date)) {
          return true;
        }
      }
    }
    return false;
  }

  protected Date findStartOfNextDay(Date date) {
    final Calendar calendar = this.createCalendar();
    calendar.setTime(date);
    calendar.add(Calendar.DATE, 1);
    calendar.set(Calendar.HOUR_OF_DAY, 0);
    calendar.set(Calendar.MINUTE, 0);
    calendar.set(Calendar.SECOND, 0);
    calendar.set(Calendar.MILLISECOND, 0);
    date = calendar.getTime();
    while (this.isHoliday(date)) {
      calendar.setTime(date);
      calendar.add(Calendar.DATE, 1);
      date = calendar.getTime();
    }
    return date;
  }

  public Calendar createCalendar() {
    return new GregorianCalendar();
  }

  protected Day findDay(final Date date) {
    final Calendar calendar = this.createCalendar();
    calendar.setTime(date);
    final int weekDayIndex = calendar.get(Calendar.DAY_OF_WEEK);
    return this.days[weekDayIndex];
  }

  protected DayPart findDayPart(final Date date) {
    DayPart dayPart = null;
    if (!this.isHoliday(date)) {
      final Day day = this.findDay(date);
      final DayPart[] dayParts = day.getDayParts();
      for (int i = 0; (i < dayParts.length) && (dayPart == null); i++) {
        final DayPart candidate = dayParts[i];
        if (candidate.includes(date)) {
          dayPart = candidate;
        }
      }
    }
    return dayPart;
  }

  protected DayPart findNextDayPart(Date date) {
    DayPart nextDayPart = null;
    while (nextDayPart == null) {
      nextDayPart = this.findDayPart(date);
      if (nextDayPart == null) {
        date = this.findStartOfNextDay(date);
        final Object[] result = new Object[2];
        final Day day = this.findDay(date);
        day.findNextDayPartStart(0, date, result);
        nextDayPart = (DayPart) result[1];
      }
    }
    return nextDayPart;
  }

  // getters and setters //////////////////////////////////////////////////////

  public long getBusinessDayInMillis() {
    return this.businessDayInMillis;
  }

  public void setBusinessDayInMillis(final long businessDayInMillis) {
    this.businessDayInMillis = businessDayInMillis;
  }

  public long getBusinessMonthInMillis() {
    return this.businessMonthInMillis;
  }

  public void setBusinessMonthInMillis(final long businessMonthInMillis) {
    this.businessMonthInMillis = businessMonthInMillis;
  }

  public long getBusinessWeekInMillis() {
    return this.businessWeekInMillis;
  }

  public void setBusinessWeekInMillis(final long businessWeekInMillis) {
    this.businessWeekInMillis = businessWeekInMillis;
  }

  public long getBusinessYearInMillis() {
    return this.businessYearInMillis;
  }

  public void setBusinessYearInMillis(final long businessYearInMillis) {
    this.businessYearInMillis = businessYearInMillis;
  }

  public long getDayInMillis() {
    return this.dayInMillis;
  }

  public void setDayInMillis(final long dayInMillis) {
    this.dayInMillis = dayInMillis;
  }

  public Day[] getDays() {
    return this.days;
  }

  public void setDays(final Day[] days) {
    this.days = days;
  }

  public Holiday[] getHolidays() {
    return this.holidays;
  }

  public void setHolidays(final Holiday[] holidays) {
    this.holidays = holidays;
  }

  public long getHourInMillis() {
    return this.hourInMillis;
  }

  public void setHourInMillis(final long hourInMillis) {
    this.hourInMillis = hourInMillis;
  }

  public long getMinuteInMillis() {
    return this.minuteInMillis;
  }

  public void setMinuteInMillis(final long minuteInMillis) {
    this.minuteInMillis = minuteInMillis;
  }

  public long getMonthInMillis() {
    return this.monthInMillis;
  }

  public void setMonthInMillis(final long monthInMillis) {
    this.monthInMillis = monthInMillis;
  }

  public long getSecondInMillis() {
    return this.secondInMillis;
  }

  public void setSecondInMillis(final long secondInMillis) {
    this.secondInMillis = secondInMillis;
  }

  public TimeZone getTimeZone() {
    return this.timeZone;
  }

  public void setTimeZone(final TimeZone timeZone) {
    this.timeZone = timeZone;
  }

  public long getWeekInMillis() {
    return this.weekInMillis;
  }

  public void setWeekInMillis(final long weekInMillis) {
    this.weekInMillis = weekInMillis;
  }

  public long getYearInMillis() {
    return this.yearInMillis;
  }

  public void setYearInMillis(final long yearInMillis) {
    this.yearInMillis = yearInMillis;
  }
}
