/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.deploy;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.ow2.orchestra.pvm.Deployment;
import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * manages sequences of deployers for multiple process languages.
 * 
 * <p>
 * Each process language has a language-id. In a deployment a language-id can be
 * given. This deployer manager will be able to find a sequence of deployers
 * based on the language-id.
 * </p>
 * 
 * <p>
 * Also, an extension can be associated to a process language. In that case the
 * deployer manager can find the deployer sequence based on the extension of the
 * deployed file name.
 * </p>
 * 
 * @author Tom Baeyens
 */
public class DeployerManager {

  private static final Log LOG = Log.getLog(DeployerManager.class.getName());

  /** maps extensions to language-ids */
  protected Map<String, String> extensions = new HashMap<String, String>();

  /** maps language-ids to a sequence of process deployers */
  protected Map<String, List<Deployer>> deployers = new HashMap<String, List<Deployer>>();

  public void deploy(Deployment deployment) {
    if (deployment == null) {
      throw new PvmException("deployment is null");
    }

    // find the language based on the extension
    String language = deployment.getLanguage();
    if (language == null) {
      String deploymentName = deployment.getName();
      LOG
          .trace("no language specified in deployment, looking for extension in "
              + deploymentName);
      if (deploymentName == null) {
        throw new PvmException("no name nor language specified in deployment");
      }
      Iterator<Entry<String, String>> iter = extensions.entrySet().iterator();
      while ((iter.hasNext()) && (language == null)) {
        Map.Entry<String, String> extensionEntry = iter.next();
        if (deploymentName.endsWith(extensionEntry.getKey())) {
          language = extensionEntry.getValue();
        }
      }

      if (language == null) {
        throw new PvmException(
            "couldn't find process language matching extension of "
                + deploymentName + " for deploying " + deployment);
      }
    }

    // find the process parser based on the specified language
    LOG.trace("looking up process deployer sequence for language " + language);
    List<Deployer> deployerSequence = deployers.get(language);
    if (deployerSequence == null) {
      throw new PvmException("couldn't find deployer sequence for language "
          + language + " for deploying " + deployment);
    }

    LOG.debug("deploying deployment " + deployment + " with " + language
        + "-parser");

    for (Deployer deployer : deployerSequence) {
      LOG.trace("deploying " + deployment + " to " + deployer);
      deployer.deploy(deployment);
    }
  }

  public static String getExtension(String fileName) {
    if ((fileName == null) || (fileName.length() == 0)) {
      throw new PvmException(
          "can't extract extension of deployment: no file name");
    }

    int lastDotIndex = fileName.lastIndexOf('.');
    if (lastDotIndex == -1) {
      lastDotIndex = 0;
    }
    return fileName.substring(lastDotIndex);
  }

  public void addLanguage(String language, String extension,
      List<Deployer> deployerSequence) {
    if (extension != null) {
      extensions.put(extension, language);
    }
    deployers.put(language, deployerSequence);
  }

  public void setExtensions(Map<String, String> extensions) {
    this.extensions = extensions;
  }

  public void setDeployers(Map<String, List<Deployer>> deployers) {
    this.deployers = deployers;
  }
}
