/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */package org.ow2.orchestra.pvm.internal.jms;

import java.util.Enumeration;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.Topic;

import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * message printing and destination name extraction.
 *
 * @author Tom Baeyens, Jim Rigsbee
 */
public final class JmsMessageUtil {

  private static final Log LOG = Log.getLog(JmsMessageUtil.class.getName());

  private JmsMessageUtil() { }

  public static String dump(final Message message) {

    final StringBuffer sb = new StringBuffer();

    sb.append("JMS MessageImpl Dump\n").append(
        "MessageImpl type is " + message.getClass().getName() + "\n");

    try {
      if (message instanceof ObjectMessage) {
        sb
            .append("MessageImpl object type is "
                + ((ObjectMessage) message).getObject().getClass().getName()
                + "\n");
      }
      sb.append("Reply to " + JmsMessageUtil.getDestinationName(message.getJMSReplyTo())
          + "\n");
      final Enumeration e = message.getPropertyNames();
      while (e.hasMoreElements()) {
        final String propertyName = (String) e.nextElement();
        final Object property = message.getObjectProperty(propertyName);
        sb.append("Property " + propertyName + " value " + property.toString()
            + "\n");
      }
    } catch (final JMSException j) {
      JmsMessageUtil.LOG.error("JMS exception while dumping message", j);
    }

    return sb.toString();
  }

  public static String getDestinationName(final Destination d) {
    try {
      if (d instanceof Queue) {
        return ((Queue) d).getQueueName();
      } else if (d instanceof Topic) {
        return ((Topic) d).getTopicName();
      }
    } catch (final JMSException j) {
      JmsMessageUtil.LOG.error("JMS exception while getting destination name", j);
    }

    return null;
  }
}
