/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.pvm.internal.jobexecutor;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Transaction;
import org.ow2.orchestra.pvm.internal.job.TimerImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.job.Timer;
import org.ow2.orchestra.pvm.session.TimerSession;

/**
 * Timers created with this service are committed at the end of the transaction,
 * so their execution will be late if the delay is shorter than the transaction.
 * In that case, they will be executed at the end of the transaction.
 *
 * @author Tom Baeyens, Pascal Verdage
 */
public class JobExecutorTimerSession implements TimerSession {

  private static final Log LOG = Log.getLog(TimerSession.class.getName());

  /* injected */
  private Transaction transaction;

  /* injected */
  private JobExecutor jobExecutor;

  /* injected */
  private JobDbSession jobDbSession;

  private boolean jobExecutorNotificationScheduled = false;

  /* injected */
  private int defaultRetries = 3;

  public void schedule(final TimerImpl timer) {
    JobExecutorTimerSession.LOG.debug("scheduling " + timer);
    this.validate(timer);

    if (timer.isDefaultRetries()) {
      timer.setRetries(this.defaultRetries);
    }

    this.jobDbSession.save(timer);
    if (!this.jobExecutorNotificationScheduled) {
      this.jobExecutorNotificationScheduled = true;
      this.transaction
          .registerSynchronization(new JobAddedNotification(this.jobExecutor));
    }
  }

  protected void validate(final Timer timer) {
    if (timer == null) {
      throw new PvmException("null timer scheduled");
    }
    if (timer.getExecution() == null) {
      throw new PvmException("timer has no execution specified");
    }
    if ((timer.getSignalName() == null) && (timer.getEventName() == null)) {
      throw new PvmException("timer has no signalName or eventName specified");
    }
    if (timer.getDueDate() == null) {
      throw new PvmException("timer scheduled at null date");
    }
    if (timer.getDueDate().getTime() < 0) {
      throw new PvmException("timer scheduled with a negative date");
    }
  }

  public void cancel(final TimerImpl timer) {
    JobExecutorTimerSession.LOG.debug("canceling " + timer);
    if (timer != null) {
      this.jobDbSession.delete(timer);
    } else {
      throw new PvmException("timer is null");
    }
  }


  /**
   * @param defaultRetries the defaultRetries to set
   */
  public void setDefaultRetries(final int defaultRetries) {
    this.defaultRetries = defaultRetries;
  }
}
