/*
  * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.wire.binding;

import java.util.List;

import org.ow2.orchestra.pvm.internal.type.Converter;
import org.ow2.orchestra.pvm.internal.type.Matcher;
import org.ow2.orchestra.pvm.internal.type.Type;
import org.ow2.orchestra.pvm.internal.type.TypeMapping;
import org.ow2.orchestra.pvm.internal.type.matcher.ClassNameMatcher;
import org.ow2.orchestra.pvm.internal.type.matcher.HibernateLongIdMatcher;
import org.ow2.orchestra.pvm.internal.type.matcher.HibernateStringIdMatcher;
import org.ow2.orchestra.pvm.internal.type.matcher.SerializableMatcher;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.descriptor.VariableTypeResolverDescriptor;
import org.ow2.orchestra.pvm.internal.xml.Parse;
import org.ow2.orchestra.pvm.internal.xml.Parser;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;

/**
 * parses a descriptor for Boolean.TRUE.
 *
 * See schema docs for more details.
 *
 * @author Tom Baeyens
 */
public class VariableTypesBinding extends WireDescriptorBinding {

  public VariableTypesBinding() {
    super("variable-types");
  }

  public Object parse(final Element element, final Parse parse, final Parser parser) {
    parser.importExternalReferences(element, parse);

    final VariableTypeResolverDescriptor variableTypeResolverDescriptor = new VariableTypeResolverDescriptor();

    final List<Element> typeElements = XmlUtil.elements(element, "type");
    if (typeElements != null) {
      for (final Element typeElement : typeElements) {
        final TypeMapping typeMapping = this.parseTypeMapping(typeElement, parse, parser);
        variableTypeResolverDescriptor.addTypeMapping(typeMapping);
      }
    }
    return variableTypeResolverDescriptor;
  }

  protected TypeMapping parseTypeMapping(final Element element, final Parse parse,
      final Parser parser) {
    final TypeMapping typeMapping = new TypeMapping();
    final Type type = new Type();
    typeMapping.setType(type);

    // type name
    if (element.hasAttribute("name")) {
      type.setName(element.getAttribute("name"));
    }

    final String hibernateSessionFactoryName = XmlUtil.attribute(element,
        "hibernate-session-factory");

    // first we get the matcher
    Matcher matcher = null;
    if (element.hasAttribute("class")) {
      final String className = element.getAttribute("class");

      // if type="serializable"
      if ("serializable".equals(className)) {
        matcher = new SerializableMatcher();

        // if type="persistable"
      } else if ("persistable".equals(className)) {
        if (element.hasAttribute("id-type")) {
          final String idType = element.getAttribute("id-type");
          if ("long".equalsIgnoreCase(idType)) {
            matcher = new HibernateLongIdMatcher(hibernateSessionFactoryName);
          } else if ("string".equalsIgnoreCase(idType)) {
            matcher = new HibernateStringIdMatcher(hibernateSessionFactoryName);
          } else {
            parse.addProblem("id-type was not 'long' or 'string': " + idType);
          }
        } else {
          parse.addProblem("id-type is required in a persistable type");
        }

        // otherwise, we expect type="some.java.ClassName"
      } else {
        matcher = new ClassNameMatcher(className);
      }

    } else {
      // look for the matcher element
      final Element matcherElement = XmlUtil.element(element, "matcher");
      final Element matcherObjectElement = XmlUtil.element(matcherElement);
      if (matcherObjectElement != null) {
        final Object matcherObject = parser.parseElement(matcherObjectElement, parse);
        try {
          matcher = (Matcher) matcherObject;
        } catch (final ClassCastException e) {
          parse.addProblem("matcher is not a " + Matcher.class.getName() + ": "
              + (matcherObject != null ? matcherObject.getClass().getName() : "null"));
        }
      } else {
        parse
            .addProblem("no matcher specified in " + XmlUtil.toString(element));
      }
    }

    typeMapping.setMatcher(matcher);

    // parsing the converter
    Converter converter = null;
    if (element.hasAttribute("converter")) {
      final String converterClassName = element.getAttribute("converter");
      final ClassLoader classLoader = parse.getClassLoader();
      try {
        final Class< ? > converterClass = ReflectUtil.loadClass(classLoader,
            converterClassName);
        converter = (Converter) converterClass.newInstance();
      } catch (final Exception e) {
        parse
            .addProblem("couldn't instantiate converter " + converterClassName);
      }
    } else {
      // look for the matcher element
      final Element converterElement = XmlUtil.element(element, "converter");
      final Element converterObjectElement = XmlUtil.element(converterElement);
      if (converterObjectElement != null) {
        final Object converterObject = parser.parseElement(converterObjectElement, parse);
        try {
          converter = (Converter) converterObject;
        } catch (final ClassCastException e) {
          parse.addProblem("converter is not a " + Converter.class.getName()
              + ": "
              + (converterObject != null ? converterObject.getClass().getName() : "null"));
        }
      }
    }

    type.setConverter(converter);

    // parsing the variable class

    Class< ? > variableClass = null;
    if (element.hasAttribute("variable-class")) {
      final String variableClassName = element.getAttribute("variable-class");
      final ClassLoader classLoader = parse.getClassLoader();
      try {
        variableClass = ReflectUtil.loadClass(classLoader, variableClassName);
      } catch (final Exception e) {
        parse.addProblem("couldn't instantiate variable-class "
            + variableClassName, e);
      }
    } else {
      parse.addProblem("variable-class is required on a type: "
          + XmlUtil.toString(element));
    }

    type.setVariableClass(variableClass);

    return typeMapping;
  }
}
