package org.ow2.orchestra.pvm.internal.wire.descriptor;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.ArrayUtil;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireDefinition;
import org.ow2.orchestra.pvm.internal.wire.WireException;
import org.ow2.orchestra.pvm.internal.wire.operation.FieldOperation;
import org.ow2.orchestra.pvm.internal.wire.operation.Operation;

/**
 * <p>
 * This {@link Descriptor} creates and initializes an object. Objects can be
 * instantiated from a constructor or from a method invocation.
 * </p>
 *
 * <p>
 * The way to create an object is specified one of these methods (see <a
 * href='#create'>creating objects</a>):
 * <ul>
 * <li>className ({@link #setClassName(String)})</li>
 * <li>factoryObjectName ({@link #setFactoryObjectName(String)})</li>
 * <li>factoryDescriptor ({@link #setFactoryDescriptor(Descriptor)})</li>
 * </ul>
 * Only one of these methods can be used.
 * </p>
 *
 * <h3 id='create'>Creating objects</h3> <h4>Creating object from a constructor</h4>
 *
 * <p>
 * This method is used when
 * <code>{@link #getClassName()}!=null && {@link #getMethodName()}==null</code>.
 * </p>
 *
 * <p>
 * The {@link #construct(WireContext)} method creates a new object from a
 * constructor matching the given arguments (specified with
 * {@link #setArgDescriptors(List)}).
 * </p>
 *
 *
 * <h4>Creating an object from a method invocation</h4>
 *
 * <p>
 * The name of the method to call is specified by the method attribute.
 * </p>
 * <ul>
 * <li>If the method is <i>static</i>, the related class is
 * {@link #getClassName()}.</li>
 * <li>If the method is an object method, the object to which the method will be
 * applied is defined by:
 * <ul>
 * <li>If <code>{@link #getFactoryObjectName()}!=null</code>: the object with the name
 * factoryObjectName will be fetched from the context.</li>
 * <li>if <code>{@link #getFactoryDescriptor()}!=null</code>: the object will be
 * created from the factory descriptor.</li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * The object returned by {@link #construct(WireContext)} is the object returned
 * by the method invocation.
 * </p>
 *
 *
 * <h3>Initializing Objects</h3> <h4>Auto Wiring</h4>
 * <p>
 * If the auto wiring is enabled for the object (
 * <code>{@link #isAutoWireEnabled()}==true</code>), the WireContext will try to
 * look for objects with the same name as the fields in the class. If it finds
 * an object with that name, and if it is assignable to the field's type, it is
 * automatically injected, without the need for explicit {@link FieldOperation}
 * that specifies the injection in the wiring xml.
 * </p>
 * <p>
 * If the auto wiring is enabled and the WireContext finds an object with the
 * name of a field, but not assignable to this field, a warning message is
 * generated.
 * </p>
 *
 * <p>
 * Auto-wiring is disabled by default.
 * </p>
 *
 * <h4>Operations</h4>
 * <p>
 * Field injection or property injection are done after the auto-wiring. For
 * more information, see {@link Operation}.
 * </p>
 *
 * <p>
 * If a field was injected by auto-wiring, its value can be overridden by
 * specifying a {@link FieldOperation} or {@link org.ow2.orchestra.pvm.internal.wire.operation.PropertyOperation} operation.
 * </p>
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher (documentation)
 *
 */
public class ObjectDescriptor extends AbstractDescriptor implements Descriptor {

  private static final long serialVersionUID = 1L;
  private static Log log = Log.getLog(ObjectDescriptor.class.getName());

  private String className = null;

  /**
   * specifies the object reference on which the method will be invoked. Either
   * className, objectName or a descriptor has to be specified.
   *
   * TODO check if this member can be replaced by a RefDescriptor in the
   * factoryDescriptor member.
   *
   * */
  private String factoryObjectName = null;

  /**
   * specifies the object on which to invoke the method. Either className,
   * objectName or a descriptor has to be specified.
   */
  private Descriptor factoryDescriptor = null;

  private String methodName = null;

  /** map to db as a component */
  private List<ArgDescriptor> argDescriptors = null;
  /** list of operations to perform during initialization. */
  private List<Operation> operations = null;

  /** True if autowiring is enabled. */
  private boolean isAutoWireEnabled = false;

  public ObjectDescriptor() {
  }

  public ObjectDescriptor(final String className) {
    this.className = className;
  }

  public ObjectDescriptor(final Class< ? > clazz) {
    this.className = clazz.getName();
  }

  /**
   * This method constructs a new Object from the ObjectDefinition. This object
   * will be created from a class constructor or from a method invocation.
   *
   * @throws WireException
   *           one of the following exception occurred:
   *           <ul>
   *           <li>if the object cannot be created (unable to load the specified
   *           class or no matching constructor found)</li>
   *           <li>if the invocation of the specified method failed</li>
   *           </ul>
   * @see ObjectDescriptor
   */
  public Object construct(final WireContext wireContext) {
    Object object = null;
    Class< ? > clazz = null;

    if (this.className != null) {
      try {
        final ClassLoader classLoader = wireContext.getClassLoader();
        clazz = ReflectUtil.loadClass(classLoader, this.className);
      } catch (final Exception e) {
        throw new WireException("couldn't create object"
            + (this.name != null ? " '" + this.name + "'" : "") + ": " + e.getMessage(),
            e);
      }

      if (this.methodName == null) {
        // plain instantiation
        try {
          final Object[] args = ObjectDescriptor.getArgs(wireContext, this.argDescriptors);
          final Constructor< ? > constructor = ReflectUtil.findConstructor(clazz,
              this.argDescriptors, args);
          if (constructor == null) {
            throw new WireException("couldn't find constructor "
                + clazz.getName() + " with args " + ArrayUtil.toString(args));
          }
          object = constructor.newInstance(args);
        } catch (final WireException e) {
          throw e;
        } catch (final Exception e) {
          throw new WireException("couldn't create object '"
              + (this.name != null ? this.name : this.className) + "': " + e.getMessage(), e);
        }
      }

    } else if (this.factoryObjectName != null) {
      // referenced factory object
      object = wireContext.get(this.factoryObjectName, false);
      if (object == null) {
        throw new WireException("can't invoke method '" + this.methodName
            + "' on null, resulted from fetching object '" + this.factoryObjectName
            + "' from this wiring environment");
      }

    } else if (this.factoryDescriptor != null) {
      // factory object descriptor
      object = wireContext.create(this.factoryDescriptor, false);
      if (object == null) {
        throw new WireException(
            "created factory object is null, can't invoke method '"
                + this.methodName + "' on it");
      }
    }

    if (this.methodName != null) {
      // method invocation on object or static method invocation in case object
      // is null
      if (object != null) {
        clazz = object.getClass();
      }
      try {
        final Object[] args = ObjectDescriptor.getArgs(wireContext, this.argDescriptors);
        final Method method = ReflectUtil.findMethod(clazz, this.methodName,
            this.argDescriptors, args);
        if (method == null) {
          // throw exception but first, generate decent message
          throw new WireException("method "
              + ReflectUtil.getSignature(this.methodName, this.argDescriptors, args)
              + " is not available on "
              + (object != null ? "object " + object + " (" + clazz.getName()
                  + ")" : "class " + clazz.getName()));
        }
        if (object == null && (!Modifier.isStatic(method.getModifiers()))) {
          // A non static method is invoked on a null object
          throw new WireException("method " + clazz.getName() + "."
              + ReflectUtil.getSignature(this.methodName, this.argDescriptors, args)
              + " is not static. It cannot be called on a null object.");
        }
        object = ReflectUtil.invoke(method, object, args);

      } catch (final WireException e) {
        throw e;
      } catch (final Exception e) {
        throw new WireException("couldn't invoke factory method " + this.methodName
            + ": " + e.getMessage(), e);
      }
    }

    return object;
  }

  /**
   * Initializes the specified object. If auto-wiring was set to
   * <code>true</code>, auto-wiring is performed (see
   * {@link #autoWire(Object, WireContext)}). Fields and properties injections
   * are then performed.
   *
   */
  @Override
  public void initialize(final Object object, final WireContext wireContext) {
    try {
      // specified operations takes precedence over auto-wiring.
      // e.g. in case there is a collision between
      // a field or property injection and an autowired value,
      // the field or property injections should win.
      // That is why autowiring is done first
      if (this.isAutoWireEnabled) {
        this.autoWire(object, wireContext);
      }
      if (this.operations != null) {
        for (final Operation operation : this.operations) {
          operation.apply(object, wireContext);
        }
      }
    } catch (final Exception e) {
      throw new WireException("couldn't initialize object '"
          + (this.name != null ? this.name : this.className) + "': " + e.getMessage(), e);
    }
  }

  @Override
  public Class< ? > getType(final WireDefinition wireDefinition) {
    if (this.className != null) {
      try {
        return ReflectUtil
            .loadClass(wireDefinition.getClassLoader(), this.className);
      } catch (final PvmException e) {
        throw new WireException("couldn't get type of '"
            + (this.name != null ? this.name : this.className) + "': " + e.getMessage(), e
            .getCause());
      }
    }

    Descriptor descriptor = null;
    if (this.factoryDescriptor != null) {
      descriptor = this.factoryDescriptor;
    } else if (this.factoryObjectName != null) {
      descriptor = wireDefinition.getDescriptor(this.factoryObjectName);
    }

    if (descriptor != null) {
      final Class< ? > factoryClass = descriptor.getType(wireDefinition);
      if (factoryClass != null) {
        final Method method = ReflectUtil.findMethod(factoryClass, this.methodName,
            this.argDescriptors, null);
        if (method != null) {
          return method.getReturnType();
        }
      }
    }

    return null;
  }

  /**
   * Auto wire object present in the context and the specified object's fields.
   *
   * @param object
   *          object on which auto-wiring is performed.
   * @param wireContext
   *          context in which the wiring objects are searched.
   */
  protected void autoWire(final Object object, final WireContext wireContext) {
    Class< ? > clazz = object.getClass();
    while (clazz != null) {
      final Field[] declaredFields = clazz.getDeclaredFields();
      if (declaredFields != null) {
        for (final Field field : declaredFields) {
          if (!Modifier.isStatic(field.getModifiers())) {
            final String fieldName = field.getName();

            Object autoWireValue = null;
            if ("environment".equals(fieldName)) {
              autoWireValue = Environment.getCurrent();

            } else if (("context".equals(fieldName))
                || ("wireContext".equals(fieldName))) {
              autoWireValue = wireContext;

            } else if (wireContext.has(fieldName)) {
              autoWireValue = wireContext.get(fieldName);

            } else {
              autoWireValue = wireContext.get(field.getType());
            }
            // if auto wire value has not been found in current context,
            // search in environment
            if (autoWireValue == null) {
              final Environment currentEnvironment = Environment.getCurrent();
              if (currentEnvironment != null) {
                autoWireValue = currentEnvironment.get(fieldName);
                if (autoWireValue == null) {
                  autoWireValue = currentEnvironment.get(field.getType());
                }
              }
            }

            if (autoWireValue != null) {
              try {
                ObjectDescriptor.log.debug("auto wiring field " + fieldName + " in " + this.name);
                ReflectUtil.set(field, object, autoWireValue);
              } catch (final PvmException e) {
                if (e.getCause() instanceof IllegalArgumentException) {
                  ObjectDescriptor.log.info("WARNING: couldn't auto wire " + fieldName
                      + " (of type " + field.getType().getName() + ") "
                      + "with value " + autoWireValue + " (of type "
                      + autoWireValue.getClass().getName() + ")");
                } else {
                  ObjectDescriptor.log.info("WARNING: couldn't auto wire " + fieldName
                      + " with value " + autoWireValue);
                }
              }
            }
          }
        }
      }
      clazz = clazz.getSuperclass();
    }
  }

  /**
   * Creates a list of arguments (objects) from a list of argument descriptors.
   *
   * @param wireContext
   *          context used to create objects.
   * @param argDescriptors
   *          list of argument descriptors.
   * @return a list of object created from the descriptors.
   * @throws Exception
   */
  public static Object[] getArgs(final WireContext wireContext,
      final List<ArgDescriptor> argDescriptors) throws Exception {
    Object[] args = null;
    if (argDescriptors != null) {
      args = new Object[argDescriptors.size()];
      for (int i = 0; i < argDescriptors.size(); i++) {
        final ArgDescriptor argDescriptor = argDescriptors.get(i);
        Object arg;
        try {
          arg = wireContext.create(argDescriptor.getDescriptor(), true);
          args[i] = arg;
        } catch (final RuntimeException e) {
          // i have made sure that the runtime exception is caught everywhere
          // the getArgs method
          // is used so that a better descriptive exception can be rethrown
          throw new Exception("couldn't create argument " + i + ": "
              + e.getMessage(), e);
        }
      }
    }
    return args;
  }

  /**
   * Adds a argument descriptor to the list of arguments descriptor to used when
   * invoking the specified method.
   *
   * @param argDescriptor
   *          argument descriptor to add.
   */
  public void addArgDescriptor(final ArgDescriptor argDescriptor) {
    if (this.argDescriptors == null) {
      this.argDescriptors = new ArrayList<ArgDescriptor>();
    }
    this.argDescriptors.add(argDescriptor);
  }

  /**
   * Adds an operation to perform during initialization.
   *
   * @param operation
   *          operation to add.
   */
  public void addOperation(final Operation operation) {
    if (this.operations == null) {
      this.operations = new ArrayList<Operation>();
    }
    this.operations.add(operation);
  }

  /** convenience method to add a type based field injection */
  public void addTypedInjection(final String fieldName, final Class< ? > type) {
    this.addInjection(fieldName, new EnvironmentTypeRefDescriptor(type));
  }

  /** add an injection based on a descriptor */
  public void addInjection(final String fieldName, final Descriptor descriptor) {
    final FieldOperation injectionOperation = new FieldOperation();
    injectionOperation.setFieldName(fieldName);
    injectionOperation.setDescriptor(descriptor);
    this.addOperation(injectionOperation);
  }

  /**
   * Gets the class name of the object to create. This name is defined only when
   * creating objects from a constructor or when invoking static methods.
   *
   * @return the name of the class of the object to create.
   */
  public String getClassName() {
    return this.className;
  }

  /**
   * Sets class name of the object to create. This name is defined only when
   * creating objects from a constructor or when invoking static methods. If
   * this name is set, the factoryObjectName and factoryDescriptor should not be
   * set.
   *
   * @see #setFactoryDescriptor(Descriptor)
   * @see #setFactoryObjectName(String)
   * @param className
   *          name of the class to use.
   */
  public void setClassName(final String className) {
    this.className = className;
  }

  /**
   * Gets the list of descriptors to use to create method arguments.
   *
   * @return list of descriptors to use to create method arguments.
   */
  public List<ArgDescriptor> getArgDescriptors() {
    return this.argDescriptors;
  }

  /**
   * Sets the list of descriptors to use to create method arguments.
   *
   * @param argDescriptors
   *          list of descriptors to use to create method arguments.
   */
  public void setArgDescriptors(final List<ArgDescriptor> argDescriptors) {
    this.argDescriptors = argDescriptors;
  }

  /**
   * Gets the list of operations to perform during initialization.
   *
   * @return list of operations to perform during initialization.
   */
  public List<Operation> getOperations() {
    return this.operations;
  }

  /**
   * Sets the list of operations to perform during initialization.
   *
   * @param operations
   *          list of operations to perform during initialization.
   */
  public void setOperations(final List<Operation> operations) {
    this.operations = operations;
  }

  /**
   * Gets the Descriptor from which the object should be created.
   *
   * @return the Descriptor from which the object should be created.
   */
  public Descriptor getFactoryDescriptor() {
    return this.factoryDescriptor;
  }

  /**
   * Sets the Descriptor from which the object should be created. If this
   * Descriptor is set, the className and factoryObjectName should not be set.
   *
   * @see #setClassName(String)
   * @see #setFactoryObjectName(String)
   * @param factoryDescriptor
   *          the Descriptor from which the object should be created.
   */
  public void setFactoryDescriptor(final Descriptor factoryDescriptor) {
    this.factoryDescriptor = factoryDescriptor;
  }

  /**
   * Gets the name of the object to get from the WireContext.
   *
   * @return name of the object to get from the WireContext.
   */
  public String getFactoryObjectName() {
    return this.factoryObjectName;
  }

  /**
   * Sets name of the object to get from the WireContext. If this name is set,
   * the className and factoryDescriptor should not be set.
   *
   * @see #setClassName(String)
   * @see #setFactoryDescriptor(Descriptor)
   * @param factoryObjectName
   *          name of the object to get from the WireContext.
   */
  public void setFactoryObjectName(final String factoryObjectName) {
    this.factoryObjectName = factoryObjectName;
  }

  /**
   * Gets the name of the method to invoke.
   *
   * @return name of the method to invoke.
   */
  public String getMethodName() {
    return this.methodName;
  }

  /**
   * Sets the name of the method to invoke.
   *
   * @param methodName
   *          name of the method to invoke.
   */
  public void setMethodName(final String methodName) {
    this.methodName = methodName;
  }

  /**
   * Checks if auto-wiring is enabled
   *
   * @return <code>true</code> if auto-wiring is enabled.
   */
  public boolean isAutoWireEnabled() {
    return this.isAutoWireEnabled;
  }

  /**
   * Enables/Disables auto wiring mode.
   *
   * @param isAutoWireEnabled
   *          <code>true</code> to enable auto-wiring.
   */
  public void setAutoWireEnabled(final boolean isAutoWireEnabled) {
    this.isAutoWireEnabled = isAutoWireEnabled;
  }
}
