/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ow2.orchestra.pvm.model.CompositeElement;
import org.ow2.orchestra.pvm.model.Node;

/**
 * @author Tom Baeyens
 */
public abstract class CompositeElementImpl extends ObservableElementImpl
    implements CompositeElement {

  private static final long serialVersionUID = 1L;

  protected List<NodeImpl> nodes;
  protected boolean hasVariableDefinitions;
  protected List<VariableDefinitionImpl> variableDefinitions;
  protected boolean hasTimerDefinitions;
  protected Set<TimerDefinitionImpl> timerDefinitions;

  protected transient Map<String, NodeImpl> nodesMap;

  // nested nodes /////////////////////////////////////////////////////////////

  /**
   * creates a nested node. Also the nested node's parent pointer will be set
   * appropriatly.
   */
  public Node createNode() {
    return this.createNode(null);
  }

  /**
   * creates a nested node with the given name. Also the nested node's parent
   * pointer will be set appropriatly.
   *
   * @param nodeName
   *          may be null.
   */
  public NodeImpl createNode(final String nodeName) {
    final NodeImpl node = new NodeImpl();
    node.setName(nodeName);
    this.addNode(node);
    return node;
  }

  public Node addNode(final NodeImpl node) {
    node.setProcessDefinition(this.processDefinition);
    if (this.nodes == null) {
      this.nodes = new ArrayList<NodeImpl>();
    }
    if (!this.nodes.contains(node)) {
      this.nodes.add(node);
    }
    this.nodesMap = null;
    return node;
  }

  /**
   * removes the given node from the nested activities. Also the node's parent
   * will be nulled. This method will do nothing if the node is null or if the
   * node is not in the list of nested activities. If the node is actually
   * removed from the list of activities, the node's source will be nulled. In
   * case this is the node that was in the activitiesMap and another node exists
   * with the same name, that node (the first) will be put in the activitiesMap
   * as a replacement for the removed node.
   */
  public boolean removeNode(final NodeImpl node) {
    if ((node != null) && (this.nodes != null)) {
      final boolean isRemoved = this.nodes.remove(node);
      if (isRemoved) {
        node.setParentNode(null);
        if (this.nodes.isEmpty()) {
          this.nodes = null;
        }
        this.nodesMap = null;
      }
      return isRemoved;
    }
    return false;
  }

  /**
   * the first nested node with the given name or null of no such node exists.
   */
  public NodeImpl getNode(final String nodeName) {
    return this.getNodesMap() != null ? this.nodesMap.get(nodeName) : null;
  }

  /** is this node present ? */
  public boolean hasNode(final String nodeName) {
    return (this.getNodesMap() != null) && (this.nodesMap.containsKey(nodeName));
  }

  public Node findNode(final String nodeName) {
    if (this.nodes != null) {
      for (final NodeImpl n : this.nodes) {
        final Node node = n.findNode(nodeName);
        if (node != null) {
          return node;
        }
      }
    }
    return null;
  }

  /**
   * the list of nested activities. Beware: the actual member is returned. No
   * copy is made.
   */
  public List<Node> getNodes() {
    return (List) this.nodes;
  }

  /**
   * the nested activities, keyed by node name. If a node with the same name
   * occurs mutltiple times, the first in the list is included in the map.
   * Activities with a null value for their name are not included in the map.
   * Beware: the actual member is returned. No copy is made.
   */
  public Map<String, Node> getNodesMap() {
    if (this.nodesMap == null) {
      this.nodesMap = NodeImpl.getNodesMap(this.nodes);
    }
    return (Map) this.nodesMap;
  }

  /** indicates if this processDefinition has nodes. */
  public boolean hasNodes() {
    return (this.nodes != null) && (!this.nodes.isEmpty());
  }

  // variable definitions /////////////////////////////////////////////////////

  public List<VariableDefinitionImpl> getVariableDefinitions() {
    if (!this.hasVariableDefinitions) {
      return Collections.emptyList();
    }
    return this.variableDefinitions;
  }

  public VariableDefinitionImpl createVariableDefinition() {
    final VariableDefinitionImpl variableDefinition = new VariableDefinitionImpl();
    if (this.variableDefinitions == null) {
      this.variableDefinitions = new ArrayList<VariableDefinitionImpl>();
    }
    this.variableDefinitions.add(variableDefinition);
    this.hasVariableDefinitions = true;
    return variableDefinition;
  }

  // timer definitions ////////////////////////////////////////////////////////

  public Set<TimerDefinitionImpl> getTimerDefinitions() {
    if (!this.hasTimerDefinitions) {
      return Collections.EMPTY_SET;
    }
    return this.timerDefinitions;
  }

  public TimerDefinitionImpl createTimerDefinition() {
    final TimerDefinitionImpl timerDefinition = new TimerDefinitionImpl();
    if (this.timerDefinitions == null) {
      this.timerDefinitions = new HashSet<TimerDefinitionImpl>();
    }
    this.timerDefinitions.add(timerDefinition);
    this.hasTimerDefinitions = true;
    return timerDefinition;
  }
}
