/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.wire;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.orchestra.pvm.internal.wire.descriptor.AbstractDescriptor;

/**
 * map of {@link Descriptor}s that serve as input for a {@link WireContext}.
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher (documentation)
 */
public class WireDefinition implements Serializable {

  private static final long serialVersionUID = 1L;

  private transient ClassLoader classLoader;
  /** maps object names to {@link Descriptor}s */
  private Map<String, Descriptor> descriptors;
  private Map<Class< ? >, String> descriptorNames;
  private final boolean useTypes = true;

  /**
   * references all objects that must eagerly initialized.
   *
   * @see WireContext
   */
  private List<String> eagerInitNames;

  public WireDefinition() {
  }

  public void addDescriptor(final Descriptor descriptor) {
    if (descriptor != null) {

      String name = descriptor.getName();

      if (this.useTypes) {
        final Class< ? > type = descriptor.getType(this);

        if (type != null) {
          if ((name == null) && (descriptor instanceof AbstractDescriptor)) {
            name = type.getName();
            ((AbstractDescriptor) descriptor).setName(type.getName());
          }

          if ((name != null)
              && ((this.descriptors == null) || (!this.descriptors.containsKey(name)))) {
            // add all superclasses and interfaces to map to this descriptor
            this.addDescriptorType(type, descriptor, name);
          }
        }
      }

      if ((name != null) && (!this.hasDescriptor(name))) {
        this.putDescriptor(name, descriptor);

        if (descriptor.isEagerInit()) {
          this.addEagerInitObjectName(name);
        }
      }

    }
  }

  void putDescriptor(final String name, final Descriptor descriptor) {
    if (this.descriptors == null) {
      this.descriptors = new HashMap<String, Descriptor>();
    }
    this.descriptors.put(name, descriptor);
  }

  void addDescriptorType(final Class< ? > type, final Descriptor descriptor,
      final String descriptorName) {
    if (type != null) {
      if (this.descriptorNames == null) {
        this.descriptorNames = new HashMap<Class< ? >, String>();
      }
      if (!this.descriptorNames.containsKey(type)) {
        this.descriptorNames.put(type, descriptorName);
      }
      this.addDescriptorType(type.getSuperclass(), descriptor, descriptorName);
      final Class< ? >[] interfaceTypes = type.getInterfaces();
      if (interfaceTypes != null) {
        for (final Class< ? > interfaceType : interfaceTypes) {
          this.addDescriptorType(interfaceType, descriptor, descriptorName);
        }
      }
    }
  }

  public String getDescriptorName(final Class< ? > type) {
    return this.descriptorNames != null ? this.descriptorNames.get(type) : null;
  }

  /**
   * the descriptor with the given name from the WireDefinition or
   * <code>null</code> if the object doesn't have a descriptor.
   */
  public Descriptor getDescriptor(final String objectName) {
    if (this.descriptors == null) {
      return null;
    }
    return this.descriptors.get(objectName);
  }

  /**
   * @return previous Descriptor associated with the given name, or null if there
   *         was no Descriptor for this name.
   */
  public synchronized Descriptor addDescriptor(final String objectName,
      final Descriptor descriptor) {
    if (this.descriptors == null) {
      this.descriptors = new HashMap<String, Descriptor>();
    }
    return this.descriptors.put(objectName, descriptor);
  }

  /**
   * @return previous Descriptor associated with the given name, or null if there
   *         was no Descriptor for this name.
   */
  public synchronized Descriptor removeDescriptor(final String objectName) {
    if (this.descriptors != null) {
      return this.descriptors.remove(objectName);
    }
    return null;
  }

  public boolean hasDescriptor(final String objectName) {
    return (this.descriptors != null) && (this.descriptors.containsKey(objectName));
  }

  public void addEagerInitObjectName(final String eagerInitObjectName) {
    if (eagerInitObjectName != null) {
      if (this.eagerInitNames == null) {
        this.eagerInitNames = new ArrayList<String>();
      }
      this.eagerInitNames.add(eagerInitObjectName);
    }
  }

  public ClassLoader getClassLoader() {
    // if there is a specific classloader specified
    if (this.classLoader != null) {
      return this.classLoader;
    }
    // otherwise, use the current environment classloader
    return Thread.currentThread().getContextClassLoader();
  }

  public Map<String, Descriptor> getDescriptors() {
    return this.descriptors;
  }

  public List<String> getEagerInitNames() {
    return this.eagerInitNames;
  }

  public void setEagerInitNames(final List<String> eagerInitNames) {
    this.eagerInitNames = eagerInitNames;
  }

  public void setDescriptors(final Map<String, Descriptor> descriptors) {
    this.descriptors = descriptors;
  }

  public void setClassLoader(final ClassLoader classLoader) {
    this.classLoader = classLoader;
  }
}
