package org.ow2.orchestra.pvm.internal.wire.descriptor;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;

import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireException;

/**
 * @author Tom Baeyens
 */
public class CollectionDescriptor extends AbstractDescriptor implements
    Descriptor {

  private static final long serialVersionUID = 1L;

  private static Log log = Log.getLog(CollectionDescriptor.class.getName());

  protected String className;
  protected List<Descriptor> valueDescriptors;
  protected boolean isSynchronized;

  protected CollectionDescriptor() {
  }

  public CollectionDescriptor(final String defaultImplClassName) {
    this.className = defaultImplClassName;
  }

  public Object construct(final WireContext wireContext) {
    Object object = null;
    try {
      // instantiate
      final ClassLoader classLoader = wireContext.getClassLoader();
      final Class< ? > clazz = ReflectUtil.loadClass(classLoader, this.className);
      object = clazz.newInstance();

      if (this.isSynchronized) {
        if (object instanceof SortedSet) {
          object = Collections.synchronizedSortedSet((SortedSet) object);
        } else if (object instanceof SortedMap) {
          object = Collections.synchronizedSortedMap((SortedMap) object);
        } else if (object instanceof Set) {
          object = Collections.synchronizedSet((Set) object);
        } else if (object instanceof Map) {
          object = Collections.synchronizedMap((Map) object);
        } else if (object instanceof List) {
          object = Collections.synchronizedList((List) object);
        } else if (object instanceof Collection) {
          object = Collections.synchronizedCollection((Collection) object);
        }
      }

    } catch (final Exception e) {
      throw new WireException("couldn't create '"
          + (this.name != null ? this.name : this.className) + "': " + e.getMessage(), e);
    }
    return object;
  }

  @Override
  public void initialize(final Object object, final WireContext wireContext) {
    final Collection<Object> collection = (Collection<Object>) object;
    try {
      if (this.valueDescriptors != null) {
        for (final Descriptor descriptor : this.valueDescriptors) {
          final Object element = wireContext.create(descriptor, true);
          CollectionDescriptor.log.trace("adding element " + element + " to collection");
          collection.add(element);
        }
      }
    } catch (final WireException e) {
      throw e;
    } catch (final Exception e) {
      throw new WireException("couldn't initialize object '"
          + (this.name != null ? this.name : this.className) + "'", e);
    }
  }

  public String getClassName() {
    return this.className;
  }

  public void setClassName(final String className) {
    this.className = className;
  }

  public List<Descriptor> getValueDescriptors() {
    return this.valueDescriptors;
  }

  public void setValueDescriptors(final List<Descriptor> valueDescriptors) {
    this.valueDescriptors = valueDescriptors;
  }

  public boolean isSynchronized() {
    return this.isSynchronized;
  }

  public void setSynchronized(final boolean isSynchronized) {
    this.isSynchronized = isSynchronized;
  }
}
