/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util.hibernate;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Properties;

import org.hibernate.HibernateException;
import org.hibernate.type.NullableType;
import org.hibernate.type.TypeFactory;
import org.hibernate.usertype.ParameterizedType;
import org.hibernate.usertype.UserType;

/**
 * @author Marc Blachon, Guillaume Porcher, Charles Souillard, Miguel Valdes, Pierre Vigneras
 */
public class GenericEnumUserType implements UserType, ParameterizedType, Serializable {

  /**
   *
   */
  private static final long serialVersionUID = -1330133661718663399L;

  private static final String DEFAULT_IDENTIFIER_METHOD_NAME = "name";

  private static final String DEFAULT_VALUE_OF_METHOD_NAME = "valueOf";

  @SuppressWarnings("unchecked")
  private Class< ? extends Enum> enumClass;

  private Class< ? > identifierType;

  private Method identifierMethod;

  private Method valueOfMethod;

  private NullableType type;

  private int[] sqlTypes;

  public void setParameterValues(final Properties parameters) {
    final String enumClassName = parameters.getProperty("enumClass");
    try {
      this.enumClass = Class.forName(enumClassName).asSubclass(Enum.class);
    } catch (final ClassNotFoundException cfne) {
      throw new HibernateException("Enum class not found", cfne);
    }

    final String identifierMethodName = parameters.getProperty("identifierMethod",
        GenericEnumUserType.DEFAULT_IDENTIFIER_METHOD_NAME);

    try {
      this.identifierMethod = this.enumClass
          .getMethod(identifierMethodName, new Class[0]);
      this.identifierType = this.identifierMethod.getReturnType();
    } catch (final Exception e) {
      throw new HibernateException("Failed to obtain identifier method", e);
    }

    this.type = (NullableType) TypeFactory.basic(this.identifierType.getName());

    if (this.type == null) {
      throw new HibernateException("Unsupported identifier type "
          + this.identifierType.getName());
    }
    this.sqlTypes = new int[] { this.type.sqlType() };

    final String valueOfMethodName = parameters.getProperty("valueOfMethod",
        GenericEnumUserType.DEFAULT_VALUE_OF_METHOD_NAME);

    try {
      this.valueOfMethod = this.enumClass.getMethod(valueOfMethodName,
          new Class[] { this.identifierType });
    } catch (final Exception e) {
      throw new HibernateException("Failed to obtain valueOf method", e);
    }
  }

  public Class< ? > returnedClass() {
    return this.enumClass;
  }

  public Object nullSafeGet(final ResultSet rs, final String[] names, final Object owner)
    throws SQLException {
    final Object identifier = this.type.get(rs, names[0]);
    if (identifier == null) {
      return null;
    }

    try {
      return this.valueOfMethod.invoke(this.enumClass, new Object[] { identifier });
    } catch (final Exception e) {
      throw new HibernateException("Exception while invoking valueOf method '"
          + this.valueOfMethod.getName() + "' of " + "enumeration class '"
          + this.enumClass + "'", e);
    }
  }

  public void nullSafeSet(final PreparedStatement st, final Object value, final int index)
    throws SQLException {
    try {
      if (value == null) {
        st.setNull(index, this.type.sqlType());
      } else {
        final Object identifier = this.identifierMethod.invoke(value, new Object[0]);
        this.type.set(st, identifier, index);
      }
    } catch (final Exception e) {
      throw new HibernateException(
          "Exception while invoking identifierMethod '"
              + this.identifierMethod.getName() + "' of " + "enumeration class '"
              + this.enumClass + "'", e);
    }
  }

  public int[] sqlTypes() {
    return this.sqlTypes;
  }

  public Object assemble(final Serializable cached, final Object owner) {
    return cached;
  }

  public Object deepCopy(final Object value) {
    return value;
  }

  public Serializable disassemble(final Object value) {
    return (Serializable) value;
  }

  public boolean equals(final Object x, final Object y) {
    return x == y;
  }

  public int hashCode(final Object x) {
    return x.hashCode();
  }

  public boolean isMutable() {
    return false;
  }

  public Object replace(final Object original, final Object target, final Object owner) {
    return original;
  }
}
