/*********************************************************************
 *       This file is part of the Mandala library source code.       *
 *                   Copyright (C) 2002                              *
 *                                                                   *
 *       Authors: eipi - eipiequalsnoone@users.sf.net               *
 *                                                                   *
 * This library is free software; you can redistribute it and/or     *
 * modify it under the terms of the GNU Lesser General Public        *
 * License as published by the Free Software Foundation; either      *
 * version 2.1 of the License, or (at your option) any later version.*
 *                                                                   *
 * This library is distributed in the hope that it will be useful,   *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU *
 * Lesser General Public License for more details.                   *
 *                                                                   *
 * You should have received a copy of the GNU Lesser General Public  *
 * License along with this library; if not, write to                 *
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,   *
 * Boston, MA  02111-1307  USA                                       *
 *********************************************************************/
package org.ow2.orchestra.util;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.AbstractMap;
import java.util.Map;
import java.util.Set;

/**
   <p>A map implementation with weak values.</p>

   <p>An entry in a WeakValueMap will automatically be removed when its value
   is no longer in ordinary use. More precisely, the presence of a mapping for
   a given value will not prevent the value from being discarded by the garbage
   collector, that is, made finalizable, finalized, and then reclaimed. When a
   value has been discarded its entry is effectively removed from the map, so
   this class behaves somewhat differently than other Map implementations.</p>


   <p>Both <code>null</code> values and the <code>null</code> key are
   supported.</p>


   <p>This class is intended primarily for use with value objects whose equals
   methods test for object identity using the == operator.</p>


   <p>The behavior of the WeakValueMap class depends in part upon the actions
   of the garbage collector, so several familiar (though not required) Map
   invariants do not hold for this class. Because the garbage collector may
   discard values at any time, a WeakValueMap may behave as though an unknown
   thread is silently removing entries. In particular, even if you synchronize
   on a WeakValueMap instance and invoke none of its mutator methods, it is
   possible for the size method to return smaller values over time, for the
   isEmpty method to return false and then true, for the containsValue method
   to return true and later false for a given value, for the get method to
   return a value for a given key but later return null, for the put method to
   return null and the remove method to return false for a key that previously
   appeared to be in the map, and for successive examinations of the key set,
   the value set, and the entry set to yield successively smaller numbers of
   elements.</p>


   <p>Each value object in a WeakValueMap is stored indirectly as the referent
   of a weak reference. Therefore a value will automatically be removed only
   after the weak references to it, both inside and outside of the map, have
   been cleared by the garbage collector.</p>

   <p>This class implementation has been highly inspired on the <a
   href="http://archive.devx.com/java/free/articles/Kabutz01/Kabutz01-1.asp"
   target="_blank">SoftHashMap</a> implementation by Dr. Heinz Kabutz.</p>

   @param <K> Key type 
 * @param <V> Value type
 * 
   @author <a href="mailto:eipiequalsnoone@users.sf.net">eipi</a>
   @version 1.0

   @since 3.0
 */
public class WeakValueMap<K, V> extends AbstractMap<K, V> {
  /** The internal Map that will hold the SoftReference. */
  private final Map<K, WeakValue<K,V>> hash;
  /** Reference queue for cleared WeakReference objects. */
  private final ReferenceQueue<V> queue = new ReferenceQueue<V>();

  /**
   * <p>Create a WeakValue map instance backed by a Map of the given type.</p>
   *  
   * @param mapType the type of the real map to use
   */
  @SuppressWarnings("unchecked")
  public WeakValueMap(final Class< ? extends Map> mapType) { 
    try {
      this.hash = mapType.newInstance();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public V get(final Object key) {
    V result = null;
    // We get the WeakReference represented by that key
    WeakReference<V> ref = hash.get(key);
    if (ref != null) {
      // From the WeakReference we get the value, which can be
      // null if it was not in the map, or it was removed in
      // the processQueue() method defined below
      result = ref.get();
      if (result == null) {
        // If the value has been garbage collected, remove the
        // entry from the Map.
        hash.remove(key);
      } 
    }
    return result;
  }

  /** We define our own subclass of WeakReference which contains
        not only the value but also the key to make it easier to find
        the entry in the Map after it's been garbage collected. */
  private static class WeakValue<K,V> extends WeakReference<V> {
    private final K key;

    WeakValue(final V k, 
        final K key,
        final ReferenceQueue< ? super V> q) {
      super(k, q);
      this.key = key;
    }
  }

  /** Here we go through the ReferenceQueue and remove garbage
        collected WeakValue objects from the Map by looking them
        up using the WeakValue.key data member. */
  @SuppressWarnings("unchecked")
  private void processQueue() {
    WeakValue sv;
    while ((sv = (WeakValue) queue.poll()) != null) {
      hash.remove(sv.key);
    }
  }

  /** Here we put the key, value pair into the Map using
        a WeakValue object. */
  @Override
  public V put(final K key, final V value) {
    processQueue(); // throw out garbage collected values first
    final WeakValue<K,V> vw = hash.put(key, new WeakValue<K,V>(value, key, queue));
    return vw == null ? null : vw.get();
  }
  @Override
  public V remove(final Object key) {
    processQueue(); // throw out garbage collected values first
    final WeakValue<K,V> vw = hash.remove(key);
    return vw == null ? null : vw.get();
  }

  @Override
  public void clear() {
    processQueue(); // throw out garbage collected values
    hash.clear();
  }
  @Override
  public int size() {
    processQueue(); // throw out garbage collected values first
    return hash.size();
  }
  @Override
  public Set<Map.Entry<K,V>> entrySet() {
    //TODO: no, no, you may NOT do that!!! GRRR
    throw new UnsupportedOperationException();
  }
}
