/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util.wsdl;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.xml.WSDLLocator;

import org.xml.sax.InputSource;

public class WSDLLocatorImpl implements WSDLLocator {

  private static Logger log = Logger.getLogger(WSDLLocatorImpl.class.getName());

  private final URL wsdlURL;

  private String latestImportURI;

  public WSDLLocatorImpl(final URL wsdlFile) {
    if (wsdlFile == null) {
      throw new IllegalArgumentException("WSDL file argument cannot be null");
    }
    this.wsdlURL = wsdlFile;
  }

  public InputSource getBaseInputSource() {
    if (WSDLLocatorImpl.log.isLoggable(Level.FINE)) {
      WSDLLocatorImpl.log.fine("getBaseInputSource [wsdlUrl=" + this.wsdlURL + "]");
    }
    try {
      final InputStream is = this.wsdlURL.openStream();
      return new InputSource(is);
    } catch (final IOException e) {
      throw new WsdlException("Cannot access wsdl from [" + this.wsdlURL
          + "], ", e);
    }
  }

  public String getBaseURI() {
    return this.wsdlURL.toExternalForm();
  }

  public InputSource getImportInputSource(final String parent, String resource) {
    if (WSDLLocatorImpl.log.isLoggable(Level.FINE)) {
      WSDLLocatorImpl.log.fine("getImportInputSource [parent=" + parent + ",resource="
          + resource + "]");
    }
    URL parentURL = null;
    try {
      parentURL = new URL(parent);
    } catch (final MalformedURLException e) {
      if (WSDLLocatorImpl.log.isLoggable(Level.FINE)) {
        WSDLLocatorImpl.log.fine("Not a valid URL: " + parent);
      }
      return null;
    }
    String wsdlImport = null;
    final String external = parentURL.toExternalForm();

    // An external URL
    if (resource.startsWith("http://") || resource.startsWith("https://")) {
      wsdlImport = resource;
    } else if (resource.startsWith("/")) {
      // Absolute path
      final String beforePath = external.substring(0, external.indexOf(parentURL
          .getPath()));
      wsdlImport = beforePath + resource;
    } else {
      // A relative path
      String parentDir = external.substring(0, external.lastIndexOf("/"));

      // remove references to current dir
      while (resource.startsWith("./")) {
        resource = resource.substring(2);
      }

      // remove references to parentdir
      while (resource.startsWith("../")) {
        parentDir = parentDir.substring(0, parentDir.lastIndexOf("/"));
        resource = resource.substring(3);
      }

      wsdlImport = parentDir + "/" + resource;
    }

    try {
      if (WSDLLocatorImpl.log.isLoggable(Level.FINE)) {
        WSDLLocatorImpl.log.fine("Trying to resolve: " + wsdlImport);
      }
      final InputSource inputSource = new InputSource(wsdlImport);
      if (WSDLLocatorImpl.log.isLoggable(Level.FINE)) {
        WSDLLocatorImpl.log.fine("Trying to resolve: inputSource = " + inputSource);
      }
      if (inputSource != null) {
        this.latestImportURI = wsdlImport;
        if (WSDLLocatorImpl.log.isLoggable(Level.FINE)) {
          WSDLLocatorImpl.log.fine("Trying to resolve: latestImportURI = " + this.latestImportURI);
        }
      } else {
        throw new IllegalArgumentException("Cannot resolve imported resource: "
            + wsdlImport);
      }
      return inputSource;
    } catch (final RuntimeException rte) {
      throw rte;
    } catch (final Exception e) {
      throw new WsdlException("Cannot access imported wsdl [" + wsdlImport
          + "], ", e);
    }
  }

  public String getLatestImportURI() {
    return this.latestImportURI;
  }

  public void close() {
    // TODO Auto-generated method stub

  }
}
