package org.ow2.orchestra.util;

import java.io.Serializable;
import java.util.concurrent.atomic.AtomicReference;

/**
 * This class provides an efficient way --- using hardware
 * instructions if available --- for holding a variable that can only
 * be initialized once (aka: final variables).
 *
 * This class helps when it is not possible to declare a variable as
 * <code>final</code> but still, you need it to be initialized only
 * once (e.g: in a setter).
 *
 * Note that you cannot initialize a variable to <code>null</code>
 * using this class.
 *
 * This class is thread safe.
 *
 * @author Pierre Vigneras
 * @date Feb 7, 2008
 * @param <T>
 */
public class Final<T> implements Serializable {
  private static final long serialVersionUID = 48705032925395418L;
  
  private final AtomicReference<T> atomicReference = new AtomicReference<T>();
  
  public String toString() {
    return isInitialized() ? getValue().toString() : "<uninitialized>";
  }

  /**
   * Return the current value of this variable. No check is performed. 
   * May return null.
   * 
   * @return  the current value of this variable. No check is performed. May return null.
   */
  protected T getValue() {
    return atomicReference.get();
  }


  /**
   * Set the value of this variable directly. No check is perform.
   * Multiple set is thus possible.
   * 
   * @param t the value to set this variable to.
   */
  protected void setValue(T t) {
    atomicReference.set(t);
  }

  /**
   * Initialize this variable to the given value.
   *
   * The given value <strong>should not</strong> be null.
   *
   * @param t the value
   * @throws IllegalArgumentException if the given value is <code>null</code>
   * @throws IllegalStateException if the variable has already been set.

   */
  public void init(final T t) {
    // Warning: do not remove this check as it guarantees the correct
    // semantic of this class. If you remove it, consequences are:
    // 1. initialized() is no more correct, you need a boolean to know if
    // the value has been set up.
    // 2. Multiple set(null) can be called and this class contract is
    // violated.
    // 3. You can set(null) and thereafter set(non-null-value) and this also
    // violate this class contract.
    Misc.checkArgsNotNull(t);
    if (!atomicReference.compareAndSet(null, t)) {
      throw new IllegalStateException("Final variable already set to: " + t);
    }
  }


  /**
   * Return the value of the initialized variable.
   *
   * @return the value of the initialized variable, or null if it 
   * has not been initialized yet.
   */
  public T get() {
    return atomicReference.get();
  }


  /**
   * Check the initialized status of this final variable.
   * @return <code>true</code> if this final variable has been initialized,
   * <code>false</code> otherwise.
   */
  public boolean isInitialized() {
    return atomicReference.get() != null;
  }
}
