/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util;

import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.util.HashMap;
import java.util.Map;

import javax.management.Attribute;
import javax.management.MBeanException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.RuntimeMBeanException;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

/**
 * @author Marc Blachon, Guillaume Porcher, Charles Souillard, Miguel Valdes, Pierre Vigneras
 */
public class MBeanInvocationHandler implements InvocationHandler {
  /**
   *
   */
  public static final int CONNECTION_TIMEOUT = 60000;

  private final ObjectName mbeanObjectName;
  private final JMXServiceURL url;
  private JMXConnector jmxc;


  public MBeanInvocationHandler(final String jmxServiceUrl, final String jmxObjectName)
    throws MalformedURLException, MalformedObjectNameException {

    this.url = new JMXServiceURL(jmxServiceUrl);
    this.mbeanObjectName = ObjectName.getInstance(jmxObjectName);


  }
  /**
   * @return the jmxc
   */
  public synchronized JMXConnector getJmxc() throws IOException {
    if (this.jmxc == null) {
      // Do not keep connection open when idle.
      // JMX server will detect this connection as idle and close the connection
      final Map<String, Object> params = new HashMap<String, Object>();
      params.put("jmx.remote.x.client.connection.check.period" , 0);
      // connect to jmx server
      this.jmxc = JMXConnectorFactory.connect(this.url, params);
    }
    return this.jmxc;
  }

  public synchronized void closeJmxc() throws IOException {
    if (this.jmxc != null) {
      final JMXConnector connector = this.jmxc;
      this.jmxc = null;
      connector.close();
    }
  }
  /*
   * (non-Javadoc)
   *
   * @see java.lang.reflect.InvocationHandler#invoke(java.lang.Object,
   *      java.lang.reflect.Method, java.lang.Object[])
   */
  public Object invoke(final Object proxy, final Method method, final Object[] args) throws Throwable {
    try {
      if (method.getName().startsWith("set") && Character.isUpperCase(method.getName().charAt(3))) {
        // Set attribute
        final String attributeName = method.getName().substring(3);
        final Attribute attribute = new Attribute(attributeName, args[0]);
        this.getJmxc().getMBeanServerConnection().setAttribute(this.mbeanObjectName, attribute);
        return null;
      } else if (args == null && method.getName().startsWith("get") && Character.isUpperCase(method.getName().charAt(3))) {
        // Get attribute
        final String attributeName = method.getName().substring(3);
        return this.getJmxc().getMBeanServerConnection().getAttribute(this.mbeanObjectName, attributeName);
      } else if (args == null && method.getName().startsWith("is") && Character.isUpperCase(method.getName().charAt(2))) {
        // Get boolean attribute
        final String attributeName = method.getName().substring(2);
        return this.getJmxc().getMBeanServerConnection().getAttribute(this.mbeanObjectName, attributeName);
      }
      // Invoke method
      final String[] signature = (args == null) ? null : new String[args.length];
      if (args != null) {
        final Class< ? >[] types = method.getParameterTypes();
        for (int i = 0; i < signature.length; i++) {
          signature[i] = types[i].getName();
        }
      }
      return this.getJmxc().getMBeanServerConnection().invoke(this.mbeanObjectName, method.getName(), args, signature);
    } catch (final IOException e) {
      this.closeJmxc();
      throw e;
    } catch (final RuntimeMBeanException e) {
      throw e.getTargetException();
    } catch (final MBeanException e) {
      throw e.getTargetException();
    }
  }
}
