/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util;


import javax.xml.soap.MessageFactory;
import javax.xml.soap.MimeHeaders;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPConnection;
import javax.xml.soap.SOAPConnectionFactory;
import javax.xml.soap.SOAPConstants;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPMessage;
import javax.xml.soap.SOAPPart;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public final class SOAPUtil {

  private static SOAPConnectionFactory conFactory;
  private static MessageFactory messagefactory;

  private SOAPUtil() {

  }

  private static synchronized SOAPConnectionFactory getSOAPConnectionFactory() {
    if (SOAPUtil.conFactory == null) {
      try {
        SOAPUtil.conFactory = SOAPConnectionFactory.newInstance();
      } catch (final Exception e) {
        throw new RuntimeException(e);
      }
    }
    return SOAPUtil.conFactory;
  }

  private static synchronized MessageFactory getMessageFactory() {
    if (SOAPUtil.messagefactory == null) {
      try {
        SOAPUtil.messagefactory = MessageFactory.newInstance(SOAPConstants.SOAP_1_1_PROTOCOL);
      } catch (final Exception e) {
        throw new RuntimeException(e);
      }
    }
    return SOAPUtil.messagefactory;
  }

  public static SOAPMessage buildSOAPMessage(final Document document, final String soapAction) {
    SOAPMessage soapMessage = null;
    try {
      soapMessage = SOAPUtil.getMessageFactory().createMessage();

      final MimeHeaders hd = soapMessage.getMimeHeaders();
      hd.addHeader("SOAPAction", soapAction);
      final SOAPPart soapPart = soapMessage.getSOAPPart();

      final SOAPEnvelope enveloppe = soapPart.getEnvelope();
      enveloppe.addNamespaceDeclaration("env", "http://schemas.xmlsoap.org/soap/envelope/");

      final SOAPBody body = enveloppe.getBody();

      body.addDocument(document);

      soapMessage.saveChanges();
    } catch (final Exception e) {
      throw new RuntimeException("Exception while building a SOAPMessage", e);
    }
    return soapMessage;
  }

  public static SOAPMessage buildDocumentSOAPMessage(final String soapAction, final Element part) {
    Document document = null;
    try {
      document = XmlUtil.getNewDocument();
      final Node importedPart = document.importNode(part, true);
      document.appendChild(importedPart);

    } catch (final Exception e) {
      throw new RuntimeException("Exception while building Rpc SOAP Message", e);
    }
    return SOAPUtil.buildSOAPMessage(document, soapAction);
  }

  public static SOAPMessage buildDocumentSOAPMessage(final String soapAction, final String part) {
    final Document doc = XmlUtil.getDocumentFromString(part);
    return SOAPUtil.buildSOAPMessage(doc, soapAction);
  }

  public static SOAPMessage call(final SOAPMessage soapMessage, final String endpoint) {
    SOAPMessage response = null;
    SOAPConnection connection = null;
    try {
      connection = SOAPUtil.getSOAPConnectionFactory().createConnection();
      response = connection.call(soapMessage, endpoint);
    } catch (final Exception e) {
      throw new RuntimeException("Exception caught while performing call to " + endpoint, e);
    } finally {
      if (connection != null) {
        try {
          connection.close();
        } catch (final Exception e) {
          // nothing
        }
      }
    }
    return response;
  }

}

