/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.Attr;
import org.w3c.dom.CharacterData;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * convenience methods to make reading org.w3c.dom models easier.
 *
 * @author Tom Baeyens
 */
public abstract class XmlUtil {

  private static final Logger LOG = Logger.getLogger(XmlUtil.class.getName());

  private static DocumentBuilderFactory documentBuilderFactory;
  private static TransformerFactory transformerFactory;
  private static ThreadLocal<Transformer> transformer = new ThreadLocal<Transformer>() {
    @Override
    protected Transformer initialValue() {
      try {
        final Transformer domTransformer = XmlUtil.getTransformerFactory().newTransformer();
        domTransformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
        domTransformer.setOutputProperty(OutputKeys.INDENT, "yes");
        domTransformer.setOutputProperty("{http://xml.apache.org/xalan}indent-amount", "2");
        return domTransformer;
      } catch (final TransformerConfigurationException e) {
        throw new RuntimeException(e);
      }
    }

    ;
  };
  private static ThreadLocal<DocumentBuilder> documentBuilder = new ThreadLocal<DocumentBuilder>() {
    @Override
    protected DocumentBuilder initialValue() {
      final DocumentBuilderFactory myDocumentBuilderFactory = XmlUtil.getDocumentBuilderFactory();
      myDocumentBuilderFactory.setNamespaceAware(true);
      myDocumentBuilderFactory.setExpandEntityReferences(false);
      try {
        return myDocumentBuilderFactory.newDocumentBuilder();
      } catch (final ParserConfigurationException e) {
        throw new RuntimeException("Exception while creating document builder " + e.getMessage(), e);
      }
    }
  };

  private XmlUtil() {
  }

  public static List<Element> elements(final Element element, final String tagName) {
    List<Element> elements = null;
    final NodeList nodeList = element.getChildNodes();
    if (nodeList != null) {
      for (int i = 0; i < nodeList.getLength(); i++) {
        final Node child = nodeList.item(i);
        if (Element.class.isAssignableFrom(child.getClass())) {
          final Element childElement = (Element) child;
          final String childTagName = XmlUtil.getTagLocalName(childElement);
          if (childTagName.equals(tagName)) {
            if (elements == null) {
              elements = new ArrayList<Element>();
            }
            elements.add(childElement);
          }
        }
      }
    }
    return elements;
  }

  public static List<Element> elements(final Element element,
                                       final Set<String> allowedTagNames) {
    List<Element> elements = null;
    final NodeList nodeList = element.getChildNodes();
    if (nodeList != null) {
      for (int i = 0; i < nodeList.getLength(); i++) {
        final Node child = nodeList.item(i);
        if (Element.class.isAssignableFrom(child.getClass())) {
          final Element childElement = (Element) child;
          final String childTagName = XmlUtil.getTagLocalName(childElement);
          if (allowedTagNames.contains(childTagName)) {
            if (elements == null) {
              elements = new ArrayList<Element>();
            }
            elements.add(childElement);
          }
        }
      }
    }
    return elements;
  }

  public static Element element(final Element element, final String tagName) {
    Element childElement = null;
    final NodeList nodeList = element.getChildNodes();
    for (int i = 0; (i < nodeList.getLength()) && (childElement == null); i++) {
      final Node child = nodeList.item(i);
      if ((Element.class.isAssignableFrom(child.getClass()))
          && (XmlUtil.getTagLocalName((Element) child)).equals(tagName)) {
        childElement = (Element) child;
      }
    }
    return childElement;
  }

  public static Element element(final Element element, final String ns, final String localName) {
    Element childElement = null;
    final NodeList nodeList = element.getChildNodes();
    for (int i = 0; (i < nodeList.getLength()) && (childElement == null); i++) {
      final Node child = nodeList.item(i);
      if ((Element.class.isAssignableFrom(child.getClass()))
          && child.getLocalName() != null
          && child.getLocalName().equals(localName) && child.getNamespaceURI() != null
          && child.getNamespaceURI().equals(ns)) {
        childElement = (Element) child;
      }
    }
    return childElement;
  }

  public static List<Element> elements(final Element element) {
    List<Element> elements = null;
    if (element != null) {
      final NodeList nodeList = element.getChildNodes();
      if ((nodeList != null) && (nodeList.getLength() > 0)) {
        elements = new ArrayList<Element>();
        for (int i = 0; i < nodeList.getLength(); i++) {
          final Node node = nodeList.item(i);
          if (node instanceof Element) {
            elements.add((Element) node);
          }
        }
      }
    }
    return elements;
  }

  public static List<Element> elements(final Element father, final String ns,
                                       final String localName) {
    final List<Element> matchingElements = new ArrayList<Element>();
    final NodeList nl = father.getChildNodes();
    for (int i = 0; i < nl.getLength(); i++) {
      final Node n = nl.item(i);
      if (n instanceof Element && n.getLocalName() != null
          && n.getLocalName().equals(localName) && n.getNamespaceURI() != null
          && n.getNamespaceURI().equals(ns)) {
        matchingElements.add((Element) n);
      }
    }
    return matchingElements;
  }

  public static List<Element> elementsQName(final Element element,
                                            final Set<QName> allowedTagNames) {
    List<Element> elements = null;
    final NodeList nodeList = element.getChildNodes();
    if (nodeList != null) {
      for (int i = 0; i < nodeList.getLength(); i++) {
        final Node child = nodeList.item(i);
        if (Element.class.isAssignableFrom(child.getClass())) {
          final Element childElement = (Element) child;
          final QName childElementQName = new QName(childElement.getNamespaceURI(),
                  childElement.getLocalName());
          if (allowedTagNames.contains(childElementQName)) {
            if (elements == null) {
              elements = new ArrayList<Element>();
            }
            elements.add(childElement);
          }
        }
      }
    }
    return elements;
  }

  public static Element element(final Element element) {
    Element onlyChild = null;
    final List<Element> elements = XmlUtil.elements(element);
    if (elements != null && !elements.isEmpty()) {
      onlyChild = elements.get(0);
    }
    return onlyChild;
  }


  public static String toString(final Node node) {
    if (node == null) {
      return null;
    }
    if (node instanceof Document) {
      if (Proxy.isProxyClass(node.getClass())) {
        final InvocationHandler ih = Proxy.getInvocationHandler(node);
        if (ih instanceof DocumentProxy) {
          final DocumentProxy dp = (DocumentProxy) ih;
          if (!dp.isInitialized()) {
            return dp.getDocumentAsString();
          }
        }
      }
    }
    final Source source = new DOMSource(node);

    final StringWriter stringWriter = new StringWriter();
    final PrintWriter printWriter = new PrintWriter(stringWriter);
    final Result result = new StreamResult(printWriter);

    Transformer domTransformer = null;
    try {
      domTransformer = XmlUtil.getTransformer();
      domTransformer.transform(source, result);
    } catch (final Exception e) {
      Misc.fastDynamicLog(XmlUtil.LOG, Level.WARNING, "couldn't transform dom element into string representation");
      if (node instanceof Element) {
        return "<" + ((Element) node).getTagName() + " ... >...</" + ((Element) node).getTagName()
               + ">";
      } else {
        return node.getTextContent();
      }
    }

    printWriter.close();

    return stringWriter.toString();
  }

  public static String getContentText(final Element element) {
    final StringBuffer buffer = new StringBuffer();
    final NodeList nodeList = element.getChildNodes();
    for (int i = 0; i < nodeList.getLength(); i++) {
      final Node node = nodeList.item(i);
      if (node instanceof CharacterData) {
        final CharacterData characterData = (CharacterData) node;
        buffer.append(characterData.getData());
      }
    }
    return buffer.toString();
  }

  public static boolean isTextOnly(final Element element) {
    boolean isTextOnly = true;
    final NodeList nodeList = element.getChildNodes();
    for (int i = 0; i < nodeList.getLength() && isTextOnly; i++) {
      if (Element.class.isAssignableFrom(nodeList.item(i).getClass())) {
        isTextOnly = false;
      }
    }
    return isTextOnly;
  }

  public static List<Attr> attributes(final Element element) {
    final NamedNodeMap attributeMap = element.getAttributes();
    if ((attributeMap == null) || (attributeMap.getLength() == 0)) {
      return null;
    }

    final List<Attr> attributes = new ArrayList<Attr>();
    for (int i = 0; i < attributeMap.getLength(); i++) {
      attributes.add((Attr) attributeMap.item(i));
    }

    return attributes;
  }

  public static String getTagLocalName(final Element element) {
    if (element == null) {
      return null;
    }
    final String localName = element.getLocalName();
    if (localName != null) {
      return localName;
    }
    return element.getTagName();
  }

  /**
   * the attribute value or null if the attribute is not present
   */
  public static String attribute(final Element element, final String attributeName) {
    if (element.hasAttribute(attributeName)) {
      return element.getAttribute(attributeName);
    } else {
      return null;
    }
  }

  public static Boolean parseBooleanValue(final String valueText) {
    if (valueText != null) {
      // if we have to check for value true
      if (("true".equals(valueText)) || ("enabled".equals(valueText))
          || ("on".equals(valueText))) {
        return Boolean.TRUE;

      } else if (("false".equals(valueText)) || ("disabled".equals(valueText))
                 || ("off".equals(valueText))) {
        return Boolean.FALSE;
      }
    }

    return null;
  }

  public static String errorMessageAttribute(final Element element,
                                             final String attributeName, final String attributeValue, final String message) {
    return "attribute <" + XmlUtil.getTagLocalName(element) + " "
           + attributeName + "=\"" + attributeValue + "\" " + message;
  }

  public static List<String> parseList(final Element element, final String singularTagName) {
    // a null value for text represents a wildcard
    String text = XmlUtil.attribute(element, singularTagName + "s");
    // so next we'll convert a '*' into the text null value, which indicates a
    // wildcard
    if ("*".equals(text)) {
      text = null;
    }
    if (element.hasAttribute(singularTagName)) {
      final String eventText = element.getAttribute(singularTagName);
      text = text == null ? eventText : text + "," + eventText;
    }
    final List<String> eventNames = XmlUtil.parseCommaSeparatedList(text);
    return eventNames;
  }

  /**
   * parses comma or space separated list. A null return value means a wildcard.
   *
   * @return List of tokens or null if the commaSeparatedListText is null, '*',
   *         or empty
   */
  public static List<String> parseCommaSeparatedList(
          final String commaSeparatedListText) {
    List<String> entries = null;
    if (commaSeparatedListText != null) {
      if (!"*".equals(commaSeparatedListText)) {
        final StringTokenizer tokenizer = new StringTokenizer(commaSeparatedListText,
                ", ");
        while (tokenizer.hasMoreTokens()) {
          if (entries == null) {
            entries = new ArrayList<String>();
          }
          entries.add(tokenizer.nextToken());
        }
      }
    }
    return entries;
  }

  public static class NamespaceValue {

    private String prefix;
    private String localPart;

    public NamespaceValue(final String prefix, final String localPart) {
      this.setPrefix(prefix);
      this.setLocalPart(localPart);
    }

    /**
     * @param prefix the prefix to set
     */
    public void setPrefix(final String prefix) {
      this.prefix = prefix;
    }

    /**
     * @return the prefix
     */
    public String getPrefix() {
      return this.prefix;
    }

    /**
     * @param localPart the localPart to set
     */
    public void setLocalPart(final String localPart) {
      this.localPart = localPart;
    }

    /**
     * @return the localPart
     */
    public String getLocalPart() {
      return this.localPart;
    }
  }

  public static NamespaceValue attributeNamespaceValue(final Element element,
                                                       final String attributeName) {
    NamespaceValue namespaceValue = null;
    final String text = XmlUtil.attribute(element, attributeName);
    if (text != null) {
      final int colonIndex = text.indexOf(':');
      if (colonIndex == -1) {
        namespaceValue = new NamespaceValue(null, text);
      } else {
        final String prefix = text.substring(0, colonIndex);
        String localPart = null;
        if (text.length() > colonIndex + 1) {
          localPart = text.substring(colonIndex + 1);
        }
        namespaceValue = new NamespaceValue(prefix, localPart);
      }
    }
    return namespaceValue;
  }

  public static QName attributeQName(final Element element, final String attributeName) {
    QName qname = null;

    final NamespaceValue namespaceValue = XmlUtil.attributeNamespaceValue(element,
            attributeName);
    final String text = XmlUtil.attribute(element, attributeName);
    if (namespaceValue != null) {
      // attribute exists
      if (namespaceValue.getPrefix() == null) {
        qname = new QName(text);
      } else {
        final String uri = element.lookupNamespaceURI(namespaceValue.getPrefix());
        if (uri == null) {
          throw new IllegalStateException("unknown prefix in qname " + text);
        } else if (namespaceValue.getLocalPart() == null) {
          throw new IllegalStateException("no local part in qname " + text);
        } else {
          qname = new QName(uri, namespaceValue.getLocalPart(),
                  namespaceValue.getPrefix());
        }
      }
    }
    return qname;
  }

  private static DocumentBuilderFactory getDocumentBuilderFactory() {
    if (XmlUtil.documentBuilderFactory == null) {
      XmlUtil.documentBuilderFactory = DocumentBuilderFactory.newInstance();
    }
    return XmlUtil.documentBuilderFactory;
  }

  public static TransformerFactory getTransformerFactory() {
    if (XmlUtil.transformerFactory == null) {
      XmlUtil.transformerFactory = TransformerFactory.newInstance();
      try {
        XmlUtil.transformerFactory.setAttribute("indent-number", 2);
      } catch (final IllegalArgumentException e) {
        // attribute not supported by implementation
        Misc.fastDynamicLog(XmlUtil.LOG, Level.FINEST, "Attribute 'indent-number' not supported by current transformer factory. "
                                                       + "Xml output may not be indented correctly.");
      }
    }
    return XmlUtil.transformerFactory;
  }

  public static QName getQNameFromString(final Element element, final String qnameAsString) {
    if (qnameAsString == null || element == null) {
      return null;
    }
    final int colonIndex = qnameAsString.indexOf(":");
    final String prefix = qnameAsString.substring(0, colonIndex);
    final String localName = qnameAsString.substring(colonIndex + 1);
    final String ns = XmlUtil.getNamespaceURI(element, prefix);
    return new QName(ns, localName, prefix);
  }

  public static String getNamespaceURI(final org.w3c.dom.Node n,
                                       final String prefix) {
    final Node prefixDeclaration = n.getAttributes().getNamedItem("xmlns:" + prefix);
    if (prefixDeclaration != null) {
      // we have found the good NameSpace
      return prefixDeclaration.getNodeValue();
    }
    // we have found the good NameSpace
    // we look for the NameSpace in the parent Node
    return XmlUtil.getNamespaceURI(n.getParentNode(), prefix);
  }

  private static Transformer getTransformer() {
    return XmlUtil.transformer.get();
  }

  private static DocumentBuilder getDocumentBuilder() {
    return XmlUtil.documentBuilder.get();
  }

  public static Document getNewDocument() {
    return XmlUtil.getDocumentBuilder().newDocument();
  }

  /**
   * This method writes a DOM document to a file.
   *
   * @param doc  - doc to write
   * @param file - dest file;
   */
  public static void writeXmlFile(final Document doc, final File file) {
    try {
      final String docAsString = XmlUtil.toString(doc);
      Misc.write(docAsString, file);
    } catch (final Exception e) {
      throw new RuntimeException("Exception caught while writing xml file", e);
    }
  }

  public static ByteArrayOutputStream getNodeAsBAOS(final Node n) {
    try {
      final ByteArrayOutputStream byteStream = new ByteArrayOutputStream(1);
      byteStream.write(XmlUtil.toString(n).getBytes());
      return byteStream;
    } catch (final Exception e) {
      throw new RuntimeException("Exception caugth when transforming node to bytestream", e);
    }
  }


  public static Document getDocumentFromString(final String s) {
    return XmlUtil.getDocumentFromString(s, true);
  }

  public static Document getDocumentFromString(String s, final boolean createProxy) {
    if (s == null) {
      return null;
    }
    if (createProxy) {
      return (Document) Proxy.newProxyInstance(
              XmlUtil.class.getClassLoader(),
              new Class[]{Document.class},
              new DocumentProxy(s));
    }
    final String prolog = "<?xml";
    if (s.startsWith(prolog)) {
      final String prologEnd = "?>";
      final int prologEndIndex = s.indexOf(prologEnd);
      s = s.substring(prologEndIndex + prologEnd.length());
    }
    try {
      final Document doc = XmlUtil.getDocumentBuilder().parse(new InputSource(new StringReader(s)));
      return doc;
    } catch (final Exception e) {
      throw new RuntimeException("Exception while building a document from string : " + s, e);
    }
  }


  public static Document getDocumentFromInputStream(final InputStream is) {
    try {
      final byte[] bytes = Misc.getAllContentFrom(is);
      return XmlUtil.getDocumentFromString(new String(bytes));
    } catch (final Exception e) {
      throw new RuntimeException("Exception while building a document from input stream : " + is, e);
    }
  }

  public static Document getDocumentFromReader(final Reader r) {
    try {
      final byte[] bytes = Misc.getAllContentFrom(r);
      return XmlUtil.getDocumentFromString(new String(bytes));
    } catch (final Exception e) {
      throw new RuntimeException("Exception while building a document from reader : " + r, e);
    }
  }

  public static Document getDocumentFromFile(final File xmlFile) {
    try {
      final byte[] bytes = Misc.getAllContentFrom(xmlFile);
      return XmlUtil.getDocumentFromString(new String(bytes));
    } catch (final Exception e) {
      throw new RuntimeException("Exception while building a document from file : " + xmlFile, e);
    }
  }

  public static Document getDocumentFromURL(final URL xmlUrl) {
    try {
      final InputStream inStream = xmlUrl.openStream();
      return XmlUtil.getDocumentBuilder().parse(inStream);
    } catch (final Exception e) {
      throw new RuntimeException("Unable to parse successfully xml url : " + xmlUrl, e);
    }
  }

  public static Element getDocumentWithOneElement(final QName elementQName) {
    final Document document = XmlUtil.getNewDocument();
    final Element rootElement = document.createElementNS(elementQName.getNamespaceURI(), elementQName.getLocalPart());
    document.appendChild(rootElement);
    return rootElement;
  }

  public static Document copyDocument(final Document src) {
    if (src == null) {
      return null;
    }
    final Document dest = XmlUtil.getNewDocument();
    if (src.getDocumentElement() != null) {
      final Node n = dest.importNode(src.getDocumentElement(), true);
      dest.appendChild(n);
    }
    return dest;
  }

  //Supplementary checks in case of DocumentType Nodes
  private static boolean areDocumentTypeEquals(final DocumentType documentType1, final DocumentType documentType2) {
    NamedNodeMap n1NamedNodeMap;
    NamedNodeMap n2NamedNodeMap;
    if (!XmlUtil.areStringEquals(documentType1.getPublicId(), documentType2.getPublicId())) {
      return false;
    }
    if (!XmlUtil.areStringEquals(documentType1.getSystemId(), documentType2.getSystemId())) {
      return false;
    }
    if (!XmlUtil.areStringEquals(documentType1.getInternalSubset(), documentType2.getInternalSubset())) {
      return false;
    }
    n1NamedNodeMap = documentType1.getEntities();
    n2NamedNodeMap = documentType2.getEntities();
    if (!XmlUtil.areNamedNodeMapNullOrDiffLength(n1NamedNodeMap, n2NamedNodeMap)) {
      return false;
    }
    if (!XmlUtil.areNamedNodeMapEquals(n1NamedNodeMap, n2NamedNodeMap)) {
      return false;
    }

    n1NamedNodeMap = documentType1.getNotations();
    n2NamedNodeMap = documentType2.getNotations();
    if (!XmlUtil.areNamedNodeMapNullOrDiffLength(n1NamedNodeMap, n2NamedNodeMap)) {
      return false;
    }
    if (!XmlUtil.areNamedNodeMapEquals(n1NamedNodeMap, n2NamedNodeMap)) {
      return false;
    }
    return true;
  }

  //Return true if the two NamedNodeMap are equals : equals element (using isEqualNode) at the same index, false otherwise
  private static boolean areNamedNodeMapEquals(final NamedNodeMap n1NamedNodeMap, final NamedNodeMap n2NamedNodeMap) {
    if (n1NamedNodeMap != null && n1NamedNodeMap.getLength() > 0) {
      Node n1Temp, n2Temp;
      int i = 0;
      for (i = 0; i < n1NamedNodeMap.getLength(); i++) {
        n1Temp = n1NamedNodeMap.item(i);
        n2Temp = n2NamedNodeMap.item(i);
        if (n2Temp == null) {
          return false;
        }
        n1Temp.normalize();
        n2Temp.normalize();
        if (!n1Temp.isEqualNode(n2Temp)) {
          return false;
        }
      }
    }
    return true;
  }

  //Return true if the two NamedNodeMap are null or have the same size, false otherwise
  private static boolean areNamedNodeMapNullOrDiffLength(final NamedNodeMap n1NamedNodeMap, final NamedNodeMap n2NamedNodeMap) {
    if ((n1NamedNodeMap != null && n2NamedNodeMap == null)
        || (n1NamedNodeMap == null && n2NamedNodeMap != null)) {
      return false;
    }
    if (n1NamedNodeMap == null) {
      return true;
    }
    int n1size = 0;
    for (int i = 0; i < n1NamedNodeMap.getLength(); i++) {
      if (!XMLConstants.XMLNS_ATTRIBUTE_NS_URI.equals(n1NamedNodeMap.item(i).getNamespaceURI())) {
        n1size++;
      }
    }
    int n2size = 0;
    for (int i = 0; i < n2NamedNodeMap.getLength(); i++) {
      if (!XMLConstants.XMLNS_ATTRIBUTE_NS_URI.equals(n2NamedNodeMap.item(i).getNamespaceURI())) {
        n2size++;
      }
    }
    return n1size == n2size;
  }

  /**
   * Test the equality of two nodes following same conditions that org.w3c.dom.Node.isEqualNode(Node n) except for prefix.
   * The two nodes prefix doesn't need to be equals.
   *
   * @param n1 First node
   * @param n2 Second node
   * @return true if the two nodes are equals, false otherwise
   */
  public static boolean areNodeEquals(final Node n1, final Node n2) {
    Misc.checkArgsNotNull(n1, n2);

    Node n1Temp, n2Temp;
    int i = 0;
    NamedNodeMap n1NamedNodeMap;
    NamedNodeMap n2NamedNodeMap;

    if (n1.getNodeType() != n2.getNodeType()) {
      return false;
    }
    if (n1.getNodeType() == Node.DOCUMENT_TYPE_NODE && !XmlUtil.areDocumentTypeEquals((DocumentType) n1, (DocumentType) n2)) {
      return false;
    }

    if (!XmlUtil.areStringEquals(n1.getLocalName(), n2.getLocalName())) {
      return false;
    }
    if (!XmlUtil.areStringEquals(n1.getNamespaceURI(), n2.getNamespaceURI())) {
      return false;
    }
    if (!XmlUtil.areStringEquals(n1.getNodeValue(), n2.getNodeValue())) {
      return false;
    }
    n1NamedNodeMap = n1.getAttributes();
    n2NamedNodeMap = n2.getAttributes();
    if (!XmlUtil.areNamedNodeMapNullOrDiffLength(n1NamedNodeMap, n2NamedNodeMap)) {
      return false;
    }

    String n1Name, n2Name;
    int j = 0;
    //Iterate the first NamedNodeMap and verify that the elements exist in the second NamedNodeMap
    if (n1NamedNodeMap != null && n1NamedNodeMap.getLength() > 0) {
      for (i = 0; i < n1NamedNodeMap.getLength(); i++) {
        boolean found = false;
        n1Temp = n1NamedNodeMap.item(i);
        if (XMLConstants.XMLNS_ATTRIBUTE_NS_URI.equals(n1Temp.getNamespaceURI())) {
          break;
        }
        if (n1Temp.getNodeName().contains(":")) {
          n1Name = n1Temp.getLocalName();
        } else {
          n1Name = n1Temp.getNodeName();
        }
        n1Temp.normalize();
        for (j = 0; !found && j < n2NamedNodeMap.getLength(); j++) {
          n2Temp = n2NamedNodeMap.item(j);
          if (n2Temp.getNodeName().contains(":")) {
            n2Name = n2Temp.getLocalName();
          } else {
            n2Name = n2Temp.getNodeName();
          }
          n2Temp.normalize();
          found = n1Temp.getNodeValue().equals(n2Temp.getNodeValue()) && XmlUtil.areStringEquals(n1Name, n2Name)
                  && XmlUtil.areStringEquals(n1Temp.getNamespaceURI(), n2Temp.getNamespaceURI());
        }
        if (!found) {
          return false;
        }
      }
    }

    final NodeList n1ChildNodes = n1.getChildNodes();
    final NodeList n2ChildNodes = n2.getChildNodes();
    if ((n1ChildNodes != null && n2ChildNodes == null) || (n1ChildNodes == null && n2ChildNodes != null)) {
      return false;
    }
    if (n1ChildNodes != null && n2ChildNodes != null) {
      if (n1ChildNodes.getLength() != n2ChildNodes.getLength()) {
        return false;
      }
      for (i = 0; i < n1ChildNodes.getLength(); i++) {
        n1Temp = n1ChildNodes.item(i);
        n2Temp = n2ChildNodes.item(i);
        if (n2Temp == null) {
          return false;
        }
        n1Temp.normalize();
        n2Temp.normalize();
        if (!XmlUtil.areNodeEquals(n1Temp, n2Temp)) {
          return false;
        }
      }
    }
    return true;
  }

  //Return true if the two strings are null or equals, false otherwise
  private static boolean areStringEquals(final String str1, final String str2) {
    if (str1 == null && str2 == null) {
      return true;
    }
    if (str1 == null) {
      return false;
    }
    if (str2 == null) {
      return false;
    }
    return str1.equals(str2);
  }


}
