/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util.wsdl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import javax.wsdl.Definition;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLLocator;
import javax.wsdl.xml.WSDLReader;
import javax.wsdl.xml.WSDLWriter;
import javax.xml.namespace.QName;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;

public final class WsdlUtil {

  private WsdlUtil() {

  }

  public static Definition readWsdl(WSDLLocator locator) {
    try {
      final WSDLReader wsdlReader = WSDLFactory.newInstance().newWSDLReader();
      wsdlReader.setFeature("javax.wsdl.verbose", false);
      wsdlReader.setFeature("javax.wsdl.importDocuments", true);
      return wsdlReader.readWSDL(locator);
    } catch (final WsdlException e) {
      throw e;
    } catch (final Exception e) {
      throw new WsdlException("Problem while reading wsdl file at URL : " + locator.getBaseURI(), e);
    }
  }

  public static Definition readWsdl(final URL wsdlURL) {
    return readWsdl(new WSDLLocatorImpl(wsdlURL));
  }

  /**
   * @param wsdlDefinitionElt
   * @return
   */
  public static Definition readWsdl(final Element wsdlDefinitionElt) {
    try {
      final WSDLReader wsdlReader = WSDLFactory.newInstance().newWSDLReader();
      wsdlReader.setFeature("javax.wsdl.verbose", false);
      wsdlReader.setFeature("javax.wsdl.importDocuments", true);
      return wsdlReader.readWSDL((String) null, wsdlDefinitionElt);
    } catch (final Exception e) {
      throw new WsdlException("Problem while reading wsdl from element : "
                              + XmlUtil.toString(wsdlDefinitionElt), e);
    }
  }

  public static String writeWsdl(final Definition wsdlDefinition) {
    final ByteArrayOutputStream baos = new ByteArrayOutputStream();
    try {
      WsdlUtil.writeWsdl(wsdlDefinition, baos);
    } finally {
      try {
        baos.close();
      } catch (final IOException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
      }
    }
    return baos.toString();
  }

  public static void writeWsdl(final Definition wsdlDefinition, final OutputStream sink) {
    try {
      final WSDLWriter wsdlReader = WSDLFactory.newInstance().newWSDLWriter();
      wsdlReader.writeWSDL(wsdlDefinition, sink);
    } catch (final Exception e) {
      throw new WsdlException("Problem while writing wsdl:", e);
    }
  }

  /**
   * Return the service that implements the given port type.
   *
   * @param portTypeQName name of the portType to search
   * @return the service that contains the given port.
   */
  public static List<Service> getServicesOfPortType(final QName portTypeQName, final Set<Definition> wsdlDefinitions) {
    final List<Service> matchinServices = new ArrayList<Service>();
    for (final Definition wsdlDefinition : wsdlDefinitions) {
      for (final Service service : (Collection<Service>) wsdlDefinition.getServices().values()) {
        for (final Port port : (Collection<Port>) service.getPorts().values()) {
          if (port.getBinding().getPortType().getQName().equals(portTypeQName)) {
            matchinServices.add(service);
          }
        }
      }
    }
    return matchinServices;
  }

}
