/**
 * Copyright (C) 2010  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util.wsdl;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import javax.wsdl.Definition;
import javax.wsdl.Import;
import javax.wsdl.Types;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.schema.Schema;
import javax.wsdl.extensions.schema.SchemaReference;
import javax.xml.XMLConstants;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * @author Guillaume Porcher
 */
public final class WsdlInlineUtil {

  private WsdlInlineUtil() {
  }


  /**
   * Creates XSD files for all schemas imported by the xml schema
   *
   * @param schema
   * @param types
   */
  @SuppressWarnings("unchecked")
  private static void inlineImportedXsds(final Schema schema, final Types types, Set<String> alreadyImportedXsds) {
    // process xsd:import
    final NodeList importsNl =
            schema.getElement().getOwnerDocument().getElementsByTagNameNS(XMLConstants.W3C_XML_SCHEMA_NS_URI, "import");
    for (int i = 0; i < importsNl.getLength(); i++) {
      final Element importElt = (Element) importsNl.item(i);
      if (importElt.hasAttribute("schemaLocation")) {
        final Map<String, List<SchemaReference>> schemaImportsMap = schema.getImports();
        final List<SchemaReference> schemaImports = schemaImportsMap.get(importElt.getAttribute("namespace"));
        for (final SchemaReference schemaImport : schemaImports) {
          if (schemaImport.getSchemaLocationURI() != null
              && schemaImport.getSchemaLocationURI().equals(importElt.getAttribute("schemaLocation"))) {
            if (!alreadyImportedXsds.contains(schemaImport.getSchemaLocationURI())) {
              alreadyImportedXsds.add(schemaImport.getSchemaLocationURI());
              WsdlInlineUtil.inlineImportedXsds(schemaImport.getReferencedSchema(), types, alreadyImportedXsds);
              types.addExtensibilityElement(schemaImport.getReferencedSchema());
            }
            schemaImport.setSchemaLocationURI(null);
            importElt.removeAttribute("schemaLocation");
          }
        }
      }
    }
    // process xsd:include
    final NodeList includeNl =
            schema.getElement().getOwnerDocument().getElementsByTagNameNS(XMLConstants.W3C_XML_SCHEMA_NS_URI, "include");
    for (int i = 0; i < includeNl.getLength(); i++) {
      final Element includeElt = (Element) includeNl.item(i);
      if (includeElt.hasAttribute("schemaLocation")) {
        for (final SchemaReference schemaReference : (List<SchemaReference>) schema.getIncludes()) {
          if (schemaReference.getSchemaLocationURI() != null
              && schemaReference.getSchemaLocationURI().equals(includeElt.getAttribute("schemaLocation"))) {
            if (!alreadyImportedXsds.contains(schemaReference.getSchemaLocationURI())) {
              alreadyImportedXsds.add(schemaReference.getSchemaLocationURI());
              WsdlInlineUtil.inlineImportedXsds(schemaReference.getReferencedSchema(), types, alreadyImportedXsds);
              types.addExtensibilityElement(schemaReference.getReferencedSchema());
            }
            schemaReference.setSchemaLocationURI(null);
            includeElt.removeAttribute("schemaLocation");
          }
        }
      }
    }
    // process xsd:redefines
    final NodeList redefinesNl =
            schema.getElement().getOwnerDocument().getElementsByTagNameNS(XMLConstants.W3C_XML_SCHEMA_NS_URI, "redefine");
    for (int i = 0; i < redefinesNl.getLength(); i++) {
      final Element redefineElt = (Element) redefinesNl.item(i);
      if (redefineElt.hasAttribute("schemaLocation")) {
        for (final SchemaReference schemaReference : (List<SchemaReference>) schema.getRedefines()) {
          if (schemaReference.getSchemaLocationURI() != null
              && schemaReference.getSchemaLocationURI().equals(redefineElt.getAttribute("schemaLocation"))) {
            if (!alreadyImportedXsds.contains(schemaReference.getSchemaLocationURI())) {
              alreadyImportedXsds.add(schemaReference.getSchemaLocationURI());
              WsdlInlineUtil.inlineImportedXsds(schemaReference.getReferencedSchema(), types, alreadyImportedXsds);
              types.addExtensibilityElement(schemaReference.getReferencedSchema());
            }
            schemaReference.setSchemaLocationURI(null);
            redefineElt.removeAttribute("schemaLocation");
          }
        }
      }
    }
  }

  /**
   * Generate a wsdl file for the definition.
   * Generate files for all imported resources of this wsdl definition.
   *
   * @param wsdlDefinition
   * @return
   */
  @SuppressWarnings("unchecked")
  public static void inlineWsdlFile(final Definition wsdlDefinition) {
    // Process wsdl imports
    final Map<String, List<Import>> imports = wsdlDefinition.getImports();
    if (!imports.isEmpty()) {
      for (final List<Import> l : imports.values()) {
        for (final Import impor : l) {
          final Definition importedDefinition = impor.getDefinition();
          WsdlInlineUtil.inlineWsdlFile(importedDefinition);
        }
      }
    }
    // Process XML Schema imports
    final Types types = wsdlDefinition.getTypes();
    if (types != null) {
      for (final ExtensibilityElement elt : new ArrayList<ExtensibilityElement>(types.getExtensibilityElements())) {
        if (elt instanceof Schema) {
          final Schema schema = (Schema) elt;
          WsdlInlineUtil.inlineImportedXsds(schema, types, new HashSet<String>());
        }
      }
    }
  }

}
