/*
 * Copyright (C) 2012 Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 */

package org.ow2.orchestra.util.wsdl;

import java.io.ByteArrayInputStream;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.wsdl.xml.WSDLLocator;
import org.ow2.orchestra.util.Misc;
import org.xml.sax.InputSource;

/**
 * @author Loic Albertin
 */
public class WSDLInMemoryLocator implements WSDLLocator {

  private static final Logger LOGGER = Logger.getLogger(WSDLInMemoryLocator.class.getName());

  private final Map<String, byte[]> wsdlRepository;

  private String latestImportUri;

  private final String wsdlUrl;
  private String importerPath;

  private String resolveResource(String resource) {
    // remove references to current dir
    while (resource.startsWith("./")) {
      resource = resource.substring(2);
    }
    String path = this.importerPath;
    // remove references to parentdir
    while (resource.startsWith("../")) {
      path = path.substring(path.indexOf("/") + 1);
      resource = resource.substring(3);
    }
    return path + resource;
  }

  public WSDLInMemoryLocator(final String wsdlUrl, final String importerPath,
                             final Map<String, byte[]> wsdlRepository) {
    this.importerPath = importerPath;
    this.wsdlUrl = this.resolveResource(wsdlUrl);

    this.wsdlRepository = wsdlRepository;
  }

  public InputSource getBaseInputSource() {
    Misc.fastDynamicLog(LOGGER, Level.FINE, "getBaseInputSource [wsdlUri=%s]", this.wsdlUrl);
    if (this.wsdlUrl.startsWith("http://") || this.wsdlUrl.startsWith("https://")) {
      // An external URL
      return new InputSource(this.wsdlUrl);
    }
    if (!this.wsdlRepository.containsKey(this.wsdlUrl)) {
      return null;
    }
    final byte[] content = this.wsdlRepository.get(this.wsdlUrl);
    return new InputSource(new ByteArrayInputStream(content));
  }

  public InputSource getImportInputSource(String parentLocation, String importLocation) {
    Misc.fastDynamicLog(LOGGER, Level.FINE, "getImportInputSource [parentLocation=%s, importLocation=%s]",
        parentLocation, importLocation);

    final InputSource inputSource;
    if (importLocation.startsWith("http://") || importLocation.startsWith("https://")) {
      // An external URL
      inputSource = new InputSource(importLocation);
    } else {
      final byte[] importContent = this.wsdlRepository.get(this.resolveResource(importLocation));
      if (importContent == null) {
        return null;
      }
      inputSource = new InputSource(new ByteArrayInputStream(importContent));
    }


    this.latestImportUri = importLocation;
    return inputSource;
  }

  public String getBaseURI() {
    Misc.fastDynamicLog(LOGGER, Level.FINE, "getBaseURI [wsdlUri=%s]", this.wsdlUrl);
    return this.wsdlUrl;
  }

  public String getLatestImportURI() {
    Misc.fastDynamicLog(LOGGER, Level.FINE, "getLatestImportURI [latestImportUri=%s]", this.latestImportUri);
    return this.latestImportUri;
  }

  public void close() {
    //Nothing to do
  }
}
