/**
 * Copyright (C) 2006  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.util.wsdl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.wsdl.Binding;
import javax.wsdl.BindingOperation;
import javax.wsdl.Definition;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.soap.SOAPBinding;
import javax.wsdl.extensions.soap.SOAPOperation;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLLocator;
import javax.wsdl.xml.WSDLReader;
import javax.wsdl.xml.WSDLWriter;
import javax.xml.namespace.QName;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;

public final class WsdlUtil {

  private WsdlUtil() {

  }

  public static Definition readWsdl(WSDLLocator locator) {
    try {
      final WSDLReader wsdlReader = WSDLFactory.newInstance().newWSDLReader();
      wsdlReader.setFeature("javax.wsdl.verbose", false);
      wsdlReader.setFeature("javax.wsdl.importDocuments", true);
      return wsdlReader.readWSDL(locator);
    } catch (final WsdlException e) {
      throw e;
    } catch (final Exception e) {
      throw new WsdlException("Problem while reading wsdl file at URL : " + locator.getBaseURI(), e);
    }
  }

  public static Definition readWsdl(final URL wsdlURL) {
    return readWsdl(new WSDLLocatorImpl(wsdlURL));
  }

  /**
   * @param wsdlDefinitionElt
   * @return
   */
  public static Definition readWsdl(final Element wsdlDefinitionElt) {
    try {
      final WSDLReader wsdlReader = WSDLFactory.newInstance().newWSDLReader();
      wsdlReader.setFeature("javax.wsdl.verbose", false);
      wsdlReader.setFeature("javax.wsdl.importDocuments", true);
      return wsdlReader.readWSDL((String) null, wsdlDefinitionElt);
    } catch (final Exception e) {
      throw new WsdlException("Problem while reading wsdl from element : "
          + XmlUtil.toString(wsdlDefinitionElt), e);
    }
  }

  public static String writeWsdl(final Definition wsdlDefinition) {
    final ByteArrayOutputStream baos = new ByteArrayOutputStream();
    try {
      WsdlUtil.writeWsdl(wsdlDefinition, baos);
    } finally {
      try {
        baos.close();
      } catch (final IOException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
      }
    }
    return baos.toString();
  }

  public static void writeWsdl(final Definition wsdlDefinition, final OutputStream sink) {
    try {
      final WSDLWriter wsdlReader = WSDLFactory.newInstance().newWSDLWriter();
      wsdlReader.writeWSDL(wsdlDefinition, sink);
    } catch (final Exception e) {
      throw new WsdlException("Problem while writing wsdl:", e);
    }
  }

  /**
   * Return the service that implements the given port type.
   *
   * @param portTypeQName name of the portType to search
   * @return the service that contains the given port.
   */
  public static List<Service> getServicesOfPortType(final QName portTypeQName, final Set<Definition> wsdlDefinitions) {
    final List<Service> matchinServices = new ArrayList<Service>();
    for (final Definition wsdlDefinition : wsdlDefinitions) {
      for (final Service service : (Collection<Service>) wsdlDefinition.getServices().values()) {
        for (final Port port : (Collection<Port>) service.getPorts().values()) {
          if (port.getBinding().getPortType().getQName().equals(portTypeQName)) {
            matchinServices.add(service);
          }
        }
      }
    }
    return matchinServices;
  }

  /**
   * Returns the first element of type clazz in a list of ExtensibilityElement
   *
   * @param list  a List of ExtensibilityElement
   * @param clazz the Class to search
   * @param <T>   clazz type
   * @return the first matching element or <code>null</code>
   */
  public static <T extends ExtensibilityElement> T getExtensibilityElement(List<? extends ExtensibilityElement> list,
                                                                           Class<T> clazz) {
    final List<T> elements = getExtensibilityElements(list, clazz);
    return elements.isEmpty() ? null : elements.get(0);
  }

  /**
   * Returns the list of elements of type clazz in a list of ExtensibilityElement
   *
   * @param list  a List of ExtensibilityElement
   * @param clazz the Class to search
   * @param <T>   clazz type
   * @return the list of matching element
   */
  public static <T extends ExtensibilityElement> List<T> getExtensibilityElements(List<? extends ExtensibilityElement> list,
                                                                                  Class<T> clazz) {
    final List<T> result = new ArrayList<T>();
    for (ExtensibilityElement extensibilityElement : list) {
      if (clazz.isAssignableFrom(extensibilityElement.getClass())) {
        result.add((T) extensibilityElement);
      }
    }
    return result;
  }

  /**
   * Retrieves SoapAction parameter in a BindingOperation
   *
   * @param bindingOperation The given binding operation
   * @return The soap action or <code>null</code> if not present
   */
  @SuppressWarnings("unchecked")
  public static String getSoapAction(BindingOperation bindingOperation) {
    final List<ExtensibilityElement> list = bindingOperation.getExtensibilityElements();
    final SOAPOperation soapOperation = WsdlUtil.getExtensibilityElement(list, SOAPOperation.class);
    if (soapOperation != null)
      return soapOperation.getSoapActionURI();

    return null;
  }

  /**
   * Retrieves Binding for a given BindingOperation
   *
   * @param definition       The WSDL definition
   * @param bindingOperation The given binding operation
   * @return The corresponding binding or <code>null</code>
   */
  @SuppressWarnings("unchecked")
  public static Binding getBindingForOperation(final Definition definition, final BindingOperation bindingOperation) {
    final Map<QName, Service> services = definition.getAllServices();

    for (Service service : services.values()) {
      final Map<QName, Port> ports = service.getPorts();
      for (Port port : ports.values()) {
        final Binding binding = port.getBinding();
        final List<BindingOperation> bindingOperations = binding.getBindingOperations();
        for (BindingOperation op : bindingOperations) {
          if (op.getName().equals(bindingOperation.getName()))
            return binding;
        }
      }
    }

    final Map<QName, Binding> bindings = definition.getAllBindings();
    for (QName qName : bindings.keySet()) {
      final Binding binding = definition.getBinding(qName);
      final List<BindingOperation> bindingOperations = binding.getBindingOperations();
      for (BindingOperation op : bindingOperations) {
        if (op.getName().equals(bindingOperation.getName()))
          return binding;
      }
    }

    return null;
  }

  /**
   * Retrieves a BindingOperation using a given operation name
   *
   * @param definition    The WSDL definition
   * @param operationName the given operation name
   * @return The corresponding binding operation or <code>null</code>
   */
  @SuppressWarnings("unchecked")
  public static BindingOperation getBindingOperation(final Definition definition, final String operationName) {
    final Map<QName, Service> services = definition.getAllServices();
    for (QName qName : services.keySet()) {
      final Service service = definition.getService(qName);
      final Map<String, Port> ports = service.getPorts();

      for (String key : ports.keySet()) {
        final Port port = service.getPort(key);
        final BindingOperation bindingOperation = port.getBinding().getBindingOperation(operationName, null, null);
        if (bindingOperation != null)
          return bindingOperation;
      }
    }

    final Map<QName, Binding> bindings = definition.getAllBindings();
    for (QName key : bindings.keySet()) {
      final Binding binding = bindings.get(key);
      final BindingOperation bindingOperation = binding.getBindingOperation(operationName, null, null);
      if (bindingOperation != null)
        return bindingOperation;
    }

    return null;
  }

  /**
   * Checks if a binding operation is using RPC style
   *
   * @param definition       The WSDL definition
   * @param bindingOperation The BindingOperation to check
   * @return Returns true if the BindingOperation or the associated Binding uses RPC style
   */
  @SuppressWarnings("unchecked")
  public static boolean isRpc(final Definition definition, final BindingOperation bindingOperation) {
    final SOAPOperation soapOperation = WsdlUtil.getExtensibilityElement(bindingOperation.getExtensibilityElements(),
        SOAPOperation.class);
    if ("rpc".equalsIgnoreCase(soapOperation.getStyle())) {
      return true;
    }
    final Binding binding = WsdlUtil.getBindingForOperation(definition, bindingOperation);
    final SOAPBinding soapBinding = WsdlUtil.getExtensibilityElement(binding.getExtensibilityElements(),
        SOAPBinding.class);
    return "rpc".equalsIgnoreCase(soapBinding.getStyle());
  }
}
