/**
 * Copyright 2010 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.commands.config.internal;

import org.apache.felix.gogo.commands.Action;
import org.apache.felix.gogo.commands.Argument;
import org.apache.felix.gogo.commands.Command;
import org.apache.felix.gogo.commands.Option;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.HandlerDeclaration;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.service.command.CommandSession;
import org.fusesource.jansi.Ansi;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;

import java.util.Arrays;
import java.util.Dictionary;
import java.util.Enumeration;

@Component
@Command(name="details",
         scope="config",
         description="Print the content of a Configurations")
@HandlerDeclaration("<sh:command xmlns:sh='org.ow2.shelbie'/>")
public class DetailsConfigurationAction implements Action {

    @Option(name = "-v",
            aliases = "--verbose",
            description = "Verbose mode: prints extra information.")
    private boolean verbose;

    @Argument(name = "pid",
              required = true,
              description = "Service (or factory) PID of the Configuration to show.")
    private String pid;

    private final ConfigurationAdmin ca;

    public DetailsConfigurationAction(@Requires ConfigurationAdmin ca) {
        this.ca = ca;
    }

    public Object execute(CommandSession session) throws Exception {

        Configuration config = ca.getConfiguration(pid, null);

        if (config != null) {

            Ansi buffer = Ansi.ansi();

            buffer.a(Ansi.Attribute.INTENSITY_BOLD);
            buffer.a(config.getPid());
            buffer.a(Ansi.Attribute.INTENSITY_BOLD_OFF);

            String location = config.getBundleLocation();
            boolean bound = location != null;

            if (config.getFactoryPid() != null) {
                buffer.a(" [factory]");
            }
            if (verbose && !bound) {
                buffer.render(" (unbound)");
            }

            buffer.newline();

            if (verbose && bound) {
                buffer.render("  bound to %s", location);
                buffer.newline();
            }

            Dictionary dict = config.getProperties();
            if (dict != null) {
                for (Enumeration e = dict.keys(); e.hasMoreElements();) {
                    String name = (String) e.nextElement();
                    Object value = dict.get(name);
                    String printable = value.toString();

                    if (value.getClass().isArray()) {
                        printable = printArray(value);
                    }
                    buffer.render("  * %s=%s", name, printable);
                    if (verbose) {
                        buffer.render(" (@|italic %s|@)", printType(value.getClass()));
                    }
                    buffer.newline();
                }
            }

            System.out.print(buffer.toString());

        }

        return null;
	}

    private String printType(final Class<?> type) {
        if (type.isArray()) {
            return printType(type.getComponentType()) + "[]";
        }
        if (Long.TYPE.equals(type)) {
            return "long";
        }
        if (Byte.TYPE.equals(type)) {
            return "byte";
        }
        if (Integer.TYPE.equals(type)) {
            return "int";
        }
        if (Float.TYPE.equals(type)) {
            return "float";
        }
        if (Character.TYPE.equals(type)) {
            return "char";
        }
        if (Double.TYPE.equals(type)) {
            return "double";
        }
        if (Short.TYPE.equals(type)) {
            return "short";
        }
        if (Boolean.TYPE.equals(type)) {
            return "boolean";
        }
        return type.getName();
    }

    private String printArray(final Object o) {
        if (o instanceof long[]) {
            return Arrays.toString((long[]) o);
        }
        if (o instanceof byte[]) {
            return Arrays.toString((byte[]) o);
        }
        if (o instanceof int[]) {
            return Arrays.toString((int[]) o);
        }
        if (o instanceof float[]) {
            return Arrays.toString((float[]) o);
        }
        if (o instanceof char[]) {
            return Arrays.toString((char[]) o);
        }
        if (o instanceof double[]) {
            return Arrays.toString((double[]) o);
        }
        if (o instanceof short[]) {
            return Arrays.toString((short[]) o);
        }
        if (o instanceof boolean[]) {
            return Arrays.toString((boolean[]) o);
        }
        if (o instanceof String[]) {
            return Arrays.toString((String[]) o);
        }

        return o.toString();

    }

}