/**
 * Copyright 2012 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.shelbie.commands.felix.internal;

import org.apache.felix.ipojo.annotations.Bind;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.Unbind;
import org.apache.felix.service.command.CommandProcessor;
import org.apache.felix.service.command.Function;
import org.apache.felix.shell.Command;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.ow2.shelbie.core.IConverterManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * @author Loris Bouzonnet
 */
@Component
public class FelixCommandManager {
    
    private static final String LEGACY_SCOPE = "legacy";

    private static Log logger = LogFactory.getLog(FelixCommandManager.class);

    private BundleContext bundleContext;

    @Requires
    private IConverterManager converterManager;

    private Map<String, ServiceRegistration> registrationByCommandName = new HashMap<String, ServiceRegistration>();

    public FelixCommandManager(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    @Bind(optional = true, aggregate = true)
    void bindCommand(final Command command) {

        String commandName = command.getName();
        if (registrationByCommandName.containsKey(commandName)) {
            logger.warn("Command already bound: " + commandName);
            return;
        }
        FelixCommandWrapper wrapper = new FelixCommandWrapper(command, converterManager);
        Properties properties = new Properties();
        properties.put(CommandProcessor.COMMAND_FUNCTION, commandName);
        properties.put(CommandProcessor.COMMAND_SCOPE, LEGACY_SCOPE);
        ServiceRegistration registration =
                bundleContext.registerService(Function.class.getName(), wrapper, properties);
        registrationByCommandName.put(commandName, registration);
    }

    @Unbind
    void unbindCommand(final Command command) {
        ServiceRegistration registration = registrationByCommandName.remove(command.getName());
        if (registration !=  null) {
            registration.unregister();
        }
    }

}
