/**
 * Copyright 2013 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.testing;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

import org.apache.felix.gogo.commands.Action;
import org.apache.felix.gogo.commands.Argument;
import org.apache.felix.gogo.commands.Option;
import org.apache.felix.service.command.CommandSession;
import org.fusesource.jansi.AnsiOutputStream;
import org.ow2.shelbie.testing.internal.CloneOutputStream;
import org.ow2.shelbie.testing.internal.MetaArgument;
import org.ow2.shelbie.testing.internal.MetaOption;

/**
 * Created with IntelliJ IDEA.
 * User: guillaume
 * Date: 13/02/13
 * Time: 10:30
 * To change this template use File | Settings | File Templates.
 */
public class ActionContainer {
    private final Action action;
    private final List<MetaOption> options = new ArrayList<MetaOption>();
    private final List<MetaArgument> arguments = new ArrayList<MetaArgument>();

    // streams used to get what has been printed
    private final ByteArrayOutputStream outputStream;
    private final ByteArrayOutputStream errorStream;

    // user input ?
    private InputStream inputStream;

    // user doesn't want to use our nice streams, allows to plug user streams.
    private PrintStream userOutputStream;
    private PrintStream userErrorStream;
    private InputStream userInputStream;


    public ActionContainer(Action action) {
        this.action = action;
        this.outputStream = new ByteArrayOutputStream();
        this.errorStream = new ByteArrayOutputStream();
        init();
    }

    public void setSystemOutputStream(PrintStream outputStream) {
        this.userOutputStream = outputStream;
    }

    public void setSystemErrorStream(PrintStream errorStream) {
        this.userErrorStream = errorStream;
    }

    public void setSystemInputStream(InputStream inputStream) {
        this.userInputStream = inputStream;
    }



    private void init() {

        Class<?> type = action.getClass();
        do {
            for (Field field : type.getDeclaredFields()) {
                Option option = field.getAnnotation(Option.class);
                if (option != null) {
                    options.add(new MetaOption(option, field));
                }
                Argument arg = field.getAnnotation(Argument.class);
                if (arg != null) {
                    arguments.add(new MetaArgument(arg, field));
                }
            }
            type = type.getSuperclass();
        } while (type != null);

    }

    public ActionContainer option(String name, Object value) {
        for (MetaOption option : options) {
            if (option.getNames().contains(name)) {
                option.inject(action, value);
            }
        }
        return this;
    }

    public ActionContainer argument(int index, Object value) {
        for (MetaArgument argument : arguments) {
            if (index == argument.getIndex()) {
                argument.inject(action, value);
            }
        }
        return this;
    }

    public Object execute(CommandSession session) throws Exception {
        for (MetaOption option : options) {
            if (!option.isValid()) {
                throw new IllegalStateException(String.format("Option %s is required%n", option.getNames().get(0)));
            }
        }
        for (MetaArgument argument : arguments) {
            if (!argument.isValid()) {
                throw new IllegalStateException(String.format("Argument %d is required%n", argument.getIndex()));
            }
        }

        PrintStream currentOut = System.out;
        PrintStream currentErr = System.err;
        InputStream currentIn = System.in;

        try {
            // change streams
            applyStreams();

            // call action
            return action.execute(session);
        } finally {
            //revert streams
            System.setOut(currentOut);
            System.setErr(currentErr);
            System.setIn(currentIn);
        }
    }

    // Apply streams on System.*
    protected void applyStreams() {

        PrintStream actionOutputStream;
        PrintStream actionErrorStream;
        InputStream actionInputStream;

        // user defined ?
        if (userOutputStream != null) {
            actionOutputStream = new PrintStream(new CloneOutputStream(outputStream, userOutputStream));
        } else {
            actionOutputStream = new PrintStream(outputStream);
        }

        // user defined ?
        if (userErrorStream != null) {
            actionErrorStream = new PrintStream(new CloneOutputStream(errorStream, userErrorStream));
        } else {
            actionErrorStream = new PrintStream(errorStream);
        }

        if (userInputStream != null) {
            actionInputStream = userInputStream;
        } else {
            actionInputStream = inputStream;
        }

        if (actionInputStream != null) {
            System.setIn(actionInputStream);
        }
        System.setOut(actionOutputStream);
        System.setErr(actionErrorStream);



    }

    public String getSystemOut() {
        return getSystemOut(true);
    }

    public String getSystemOut(boolean ansiSupport) {
        String content = null;
        try {
            content = outputStream.toString("UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new IllegalStateException("Cannot find encoding", e);
        }
        // remove ansi
        if (!ansiSupport) {
            return removeAnsi(content);
        }
        return content;

    }

    private String removeAnsi(final String content) {
        if (content == null) {
            return null;
        }
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            AnsiOutputStream aos = new AnsiOutputStream(baos);
            aos.write(content.getBytes());
            aos.flush();
            return baos.toString();
        } catch (IOException e) {
            return content;
        }
    }


    public String getSystemErr() {
        return getSystemErr(true);
    }

    public String getSystemErr(boolean ansiSupport) {
        String content = null;
        try {
            content = errorStream.toString("UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new IllegalStateException("Cannot find encoding", e);
        }
        // remove ansi
        if (!ansiSupport) {
            return removeAnsi(content);
        }
        return content;

    }


    public void setSystemIn(String lines) {
        this.inputStream = new ByteArrayInputStream(lines.getBytes());
    }



}
