/**
 * Copyright 2012 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.commands.builtin.internal;

import org.apache.felix.gogo.commands.Action;
import org.apache.felix.gogo.commands.Argument;
import org.apache.felix.gogo.commands.Command;
import org.apache.felix.gogo.commands.Option;
import org.apache.felix.ipojo.annotations.Bind;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.HandlerDeclaration;
import org.apache.felix.service.command.CommandSession;
import org.ow2.shelbie.core.registry.CommandFilter;
import org.ow2.shelbie.core.registry.CommandRegistry;
import org.ow2.shelbie.core.registry.info.ArgumentInfo;
import org.ow2.shelbie.core.registry.info.CommandInfo;
import org.ow2.shelbie.core.registry.info.OptionInfo;
import org.ow2.shelbie.core.registry.info.ParameterInfo;
import org.ow2.shelbie.core.registry.info.ScopeInfo;

import java.util.Collection;

/**
 * Help for commands
 */
@Component
@Command(name = "help",
         scope = "shelbie",
         description = "Display command's help")
@HandlerDeclaration("<sh:command xmlns:sh='org.ow2.shelbie' />")
public class HelpAction implements Action {

    private CommandRegistry registry;

    @Option(name = "-l",
            aliases = "--list",
            description = "Output command list as a simple list (no decorations)")
    private boolean listing = false;

    @Argument(description = "Command name (may be scoped or un-scoped)")
    private String commandName;

    public Object execute(final CommandSession session) throws Exception {

        if (commandName == null) {

            // List all the commands

            int size = 0;
            for (ScopeInfo scopeInfo : registry.getScopes()) {
                Collection<CommandInfo> commands = scopeInfo.getCommands();
                size += commands.size();
                if (!commands.isEmpty()) {
                    if (!listing) {
                        System.out.printf("%nScope '%s' (%d commands)%n", scopeInfo.getName(), commands.size());
                        System.out.printf("-----------------------------------%n");
                    }
                    for (CommandInfo command : commands) {
                        String qualifiedName = String.format("%s:%s", command.getScope(), command.getName());
                        if (command.getDescription() == null) {
                            System.out.printf("  %-40s%n", qualifiedName);
                        } else {
                            System.out.printf("  %-40s %s%n", qualifiedName, command.getDescription());
                        }
                    }
                }
            }

            if (!listing) {
                System.out.printf("%n===================================%n");
                System.out.printf("Total: %d scopes for %d commands.%n", registry.getScopes().size(), size);
            }
        } else {
            String[] fragments = commandName.split(":");
            String scope = (fragments.length == 1) ? null : fragments[0];
            String name =  (fragments.length == 1) ? fragments[0] : fragments[1];
            // Select a subset of the commands
            CommandFilter filter = (scope == null) ? new UnQualifiedCommandFilter(name) : new QualifiedCommandFilter(scope, name);

            Collection<? extends CommandInfo> commands = registry.getCommands(filter);
            for (CommandInfo command : commands) {
                printCommandDetails(command);
                System.out.printf("%n-----------------------------------%n");
            }

        }

        return null;
    }

    private void printCommandDetails(CommandInfo command) {
        String cmdDescription = command.getDescription();
        if (isEmpty(cmdDescription)) {
            cmdDescription = "No description available";
        }
        System.out.printf("%nName%n");
        System.out.printf("  %s - %s%n%n", command.getName(), cmdDescription);

        System.out.printf("Syntax%n");
        String parameters = (command.getOptions().isEmpty()) ? "" : "[option(s)]...";
        parameters += (command.getArguments().isEmpty()) ? "" : " [argument(s)]...";
        System.out.printf("  %s:%s %s%n", command.getScope(), command.getName(), parameters);

        if (!command.getOptions().isEmpty()) {
            System.out.printf("%nOption(s)%n");
            for (OptionInfo optionInfo : command.getOptions()) {
                printOption(optionInfo);
            }
        }
        if (!command.getArguments().isEmpty()) {
            System.out.printf("%nArgument(s)%n");
            // Expect the arguments to be sorted per-index
            for (ArgumentInfo argumentInfo : command.getArguments()) {
                printArgument(argumentInfo);
            }
        }
    }

    private void printArgument(ArgumentInfo argumentInfo) {
        System.out.printf("  index: %d ", argumentInfo.getIndex());
        printParameter(argumentInfo);
    }

    private void printOption(OptionInfo optionInfo) {
        // First line is meta from the option
        System.out.printf("  ");
        for (String name : optionInfo.getNames()) {
            System.out.printf("%s ", name);
        }

        printParameter(optionInfo);
    }

    private void printParameter(ParameterInfo parameterInfo) {
        if (parameterInfo.isRequired()) {
            System.out.printf("[required] ");
        }
        if (parameterInfo.isMultiValued()) {
            System.out.printf("[multi-valued] ");
        }
        System.out.printf("%n");

        // Then the expected type
        printType(parameterInfo);

        // Then the default value (if any)
        if (parameterInfo.getDefault() != null) {
            System.out.printf("    default: %s%n", parameterInfo.getDefault().toString());
        }

        // Then the textual description itself (if any)
        String description = parameterInfo.getDescription();
        if (!isEmpty(description)) {
            System.out.printf("    %s%n", description);
        }
    }

    private void printType(ParameterInfo parameterInfo) {
        Class<?> pType = parameterInfo.getType();
        String type = pType.getName();
        if (pType.isArray()) {
            type = pType.getComponentType().getName() + "[]";
        }

        System.out.printf("    type: %s%n", type);
    }

    private static boolean isEmpty(String value) {
        return (value == null) || value.isEmpty();
    }

    @Bind
    public void bindRegistry(CommandRegistry registry) {
        this.registry = registry;
    }

    private static class QualifiedCommandFilter implements CommandFilter {
        private final String scope;
        private final String name;

        public QualifiedCommandFilter(String scope, String name) {
            this.scope = scope;
            this.name = name;
        }

        public boolean accept(CommandInfo info) {
            return scope.equals(info.getScope()) && name.equals(info.getName());
        }
    }

    private static class UnQualifiedCommandFilter implements CommandFilter {
        private final String name;

        public UnQualifiedCommandFilter(String name) {
            this.name = name;
        }

        public boolean accept(CommandInfo info) {
            return name.equals(info.getName());
        }
    }
}