/**
 * Copyright 2012-2013 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.commands.builtin.internal;

import org.apache.felix.gogo.commands.Action;
import org.apache.felix.gogo.commands.Command;
import org.apache.felix.gogo.commands.Option;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.HandlerDeclaration;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.service.command.CommandSession;
import org.fusesource.jansi.Ansi;
import org.ow2.shelbie.core.ExitSessionException;
import org.ow2.shelbie.core.system.SystemService;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

import static org.fusesource.jansi.Ansi.Attribute.BLINK_OFF;
import static org.fusesource.jansi.Ansi.Attribute.BLINK_SLOW;
import static org.fusesource.jansi.Ansi.Attribute.INTENSITY_BOLD;
import static org.fusesource.jansi.Ansi.Attribute.INTENSITY_BOLD_OFF;
import static org.fusesource.jansi.Ansi.Attribute.ITALIC;
import static org.fusesource.jansi.Ansi.Attribute.ITALIC_OFF;

/**
 * @author Loris Bouzonnet
 */
@Component
@Command(name = "shutdown",
        scope = "shelbie",
        description = "Shutdown the OSGi platform")
@HandlerDeclaration("<sh:command xmlns:sh='org.ow2.shelbie'/>")
public class ShutdownAction implements Action {

    @Option(name = "-f",
            aliases = "--force",
            description = "Do not ask for confirmation")
    private boolean force = false;

    @Option(name = "-q",
            aliases = "--quiet",
            description = "Quietly shutdown (avoid traces except confirmation messages)")
    private boolean quiet = false;

    @Requires
    private SystemService system;

    private static enum Confirmation {
        YES("y", "yes"), NO("n", "no");
        private final String[] values;

        Confirmation(String... values) {
            this.values = values;
        }

        public boolean match(String input) {
            for (String value : values) {
                if (value.equalsIgnoreCase(input)) {
                    return true;
                }
            }
            return false;
        }
    }

    public Object execute(CommandSession session) throws Exception {

        if (!force && isProduction(session)) {
            printWarningMessage();
            printConfirmationMessage();

            // Expects either:
            // * 'y' or 'N' char (ignoring case) followed by '\n'
            // * '\n'
            Confirmation confirmation = Confirmation.NO;
            try {
                BufferedReader reader = new BufferedReader(new InputStreamReader(session.getKeyboard()));
                confirmation = getConfirmation(reader.readLine());
                System.out.printf("%n");
            } catch (IOException e) {
                System.out.printf("%nCannot read answer: %s%n", e.getMessage());
                System.out.flush();
            }

            if (confirmation == Confirmation.NO) {
                System.out.printf("Aborting shutdown%n");
                System.out.flush();
                return null;
            }
        }

        if (!quiet) {
            System.out.printf("The system is going down for halt NOW!%n");
            System.out.flush();
        }

        system.shutdown();

        // Close current shell
        throw new ExitSessionException();
    }

    private void printConfirmationMessage() {
        Ansi buffer = Ansi.ansi();
        buffer.a(ITALIC);
        buffer.a("Are you sure ? ");
        buffer.a(ITALIC_OFF);
        buffer.a("[y/");
        buffer.a(BLINK_SLOW);
        buffer.a("N");
        buffer.a(BLINK_OFF);
        buffer.a("] ");
        System.out.print(buffer);
        System.out.flush();
    }

    private void printWarningMessage() {
        Ansi buffer = Ansi.ansi();
        buffer.a(INTENSITY_BOLD);
        buffer.a("Shutdown will halt the entire OSGi system.");
        buffer.a(INTENSITY_BOLD_OFF);
        buffer.newline();
        System.out.print(buffer);
        System.out.flush();
    }

    private Confirmation getConfirmation(String input) {
        for (Confirmation matcher : Confirmation.values()) {
            if (matcher.match(input)) {
                return matcher;
            }
        }
        return Confirmation.NO;
    }

    private boolean isProduction(CommandSession session) {
        // TODO Have a flag in the session
        return true;
    }

}
