/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.ow2.shelbie.commands.builtin.internal.gogo;

import org.apache.felix.gogo.shell.Builtin;
import org.apache.felix.gogo.shell.Posix;
import org.apache.felix.gogo.shell.Procedural;
import org.apache.felix.gogo.shell.Shell;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.Validate;
import org.apache.felix.service.command.CommandProcessor;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;

import java.util.Dictionary;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;

/**
 * A {@code GogoShellCommands} is responsible to register/un-register a subset of
 * Gogo shell commands provided by the {@literal org.apache.felix.gogo.shell} bundle.
 *
 * @author Guillaume Sauthier
 */
@Component
public class GogoShellCommands {

    private BundleContext bundleContext;

    private Set<ServiceRegistration> regs = new HashSet<ServiceRegistration>();

    @Requires
    private CommandProcessor processor;

    public GogoShellCommands(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    @Validate
    public void start() {
        Dictionary<String, Object> dict = new Hashtable<String, Object>();
        dict.put(CommandProcessor.COMMAND_SCOPE, "gogo");

        // register commands
        dict.put(CommandProcessor.COMMAND_FUNCTION,
                 strings( "format", "getopt", "new", "set", "tac", "type"));
        regs.add(bundleContext.registerService(Builtin.class.getName(), new Builtin(), dict));

        // each provided by shelbie
        dict.put(CommandProcessor.COMMAND_FUNCTION,
                 strings(/*"each", */"if", "not", "throw", "try", "until", "while"));
        regs.add(bundleContext.registerService(Procedural.class.getName(), new Procedural(), dict));

        // echo and grep provided by shelbie
        dict.put(CommandProcessor.COMMAND_FUNCTION,
                 strings("cat"/*, "echo", "grep"*/));
        regs.add(bundleContext.registerService(Posix.class.getName(), new Posix(), dict));

        dict.put(CommandProcessor.COMMAND_FUNCTION,
                 strings("gosh", "sh", "source"));
        regs.add(bundleContext.registerService(Shell.class.getName(), new Shell(bundleContext, processor), dict));

    }

    @Invalidate
    public void stop() {
        for (ServiceRegistration reg : regs) {
            reg.unregister();
        }
    }

    private static String[] strings(final String... args) {
        return args;
    }
}
