/**
 * Copyright 2012-2013 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.console;

import jline.Terminal;
import jline.console.ConsoleReader;
import jline.console.UserInterruptException;
import jline.console.completer.AggregateCompleter;
import jline.console.completer.Completer;
import jline.console.history.FileHistory;
import jline.console.history.History;
import org.apache.felix.service.command.CommandProcessor;
import org.apache.felix.service.command.CommandSession;
import org.apache.felix.service.command.Converter;
import org.fusesource.jansi.Ansi;
import org.fusesource.jansi.AnsiRenderer;
import org.ow2.shelbie.core.ExitSessionException;
import org.ow2.shelbie.core.branding.BrandingService;
import org.ow2.shelbie.core.console.util.Streams;
import org.ow2.shelbie.core.internal.handler.completer.ScopeCompleter;
import org.ow2.shelbie.core.internal.handler.completer.ShelbieCandidateListCompletionHandler;
import org.ow2.shelbie.core.prompt.PromptService;
import org.ow2.shelbie.core.prompt.Variables;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Map;

import static java.lang.String.format;

/**
 * Created by IntelliJ IDEA.
 * User: sauthieg
 * Date: 6 janv. 2010
 * Time: 17:46:33
 * To change this template use File | Settings | File Templates.
 */
public class JLineConsole implements Runnable {

    /**
     * OSGi Shell session.
     */
    private CommandSession session;

    /**
     * JLine reader (enable ANSI support for shell).
     */
    private ConsoleReader reader;

    private boolean running = true;

    private PromptService promptService;

    private Variables sessionVariables = new SessionVariables();

    private FileHistory history;

    /**
     * Optional service used to customize the new session.
     */
    private BrandingService brandingService;

    /**
     * Exit callback to be invoked when the run() loop exits (graceful shutdown).
     * May be null.
     */
    private Runnable callback;

    public JLineConsole(final CommandProcessor processor,
                        final Completer completer,
                        final InputStream in,
                        final PrintStream out,
                        final PrintStream err,
                        final Terminal terminal) throws Exception {

        // Wrap the output and error streams to enable ANSI support
        PrintStream wrappedOut = Streams.wrap(out);
        PrintStream wrappedErr = Streams.wrap(err);

        // Create the JLine reader
        reader = new ConsoleReader(in, wrappedOut, terminal);
        reader.setHandleUserInterrupt(true);
        // We use our own completion handler to support the scopes
        reader.setCompletionHandler(new ShelbieCandidateListCompletionHandler());

        // Create an OSGi Shell session using the given streams
        session = processor.createSession(in, wrappedOut, wrappedErr);

        // Create a composite completer that use the given Completer and wrap it
        // in a Completer that tries to complement using the known scopes (from
        // SCOPE session variable)
        Completer composite = new AggregateCompleter(completer, new ScopeCompleter(completer, session));
        reader.addCompleter(composite);

        // Store the JLine reader as shell variable (just in case someone need it)
        session.put(ConsoleReader.class.getName(), reader);
    }

    public void setCallback(Runnable callback) {
        this.callback = callback;
    }

    public void setPromptService(PromptService promptService) {
        this.promptService = promptService;
    }

    public void setHistoryFile(File historyFile) {
        if (historyFile != null) {
            try {
                history = new FileHistory(historyFile);
                reader.setHistory(history);
                session.put(History.class.getName(), history);
            } catch (IOException e) {
                // Ignore
            }
        }
    }

    public void setBrandingService(BrandingService brandingService) {
        this.brandingService = brandingService;
    }

    public CommandSession getSession() {
        return session;
    }

    public void close() {
        if (history != null) {
            try {
                history.flush();
            } catch (IOException e) {

            }
        }
        running = false;
        reader.shutdown();
    }

    public void run() {
        if (brandingService != null) {
            doSessionBranding();
        }
        while (running) {

            try {
                String rendered = getRenderedPrompt();
                String line = reader.readLine(rendered);

                // a null return signify that the user wanted to exit (^D)
                if (line == null) {
                    session.getConsole().printf("%n^D%n");
                    running = false;
                    continue;
                }

                // Execute the command line
                Object result = session.execute(line);
                session.put(Constants.LAST_RESULT_VARIABLE, result); // set $_ to last result

                // Format the result (if any)
                if (result != null) {
                    CharSequence value = session.format(result, Converter.INSPECT);
                    session.getConsole().println(value);
                }
            } catch (ExitSessionException e) {
                running = false;
            } catch (UserInterruptException uie) {
                // User hit control-c
                // Simply kill the current line
                session.getConsole().println("^C");
            } catch (IOException ioe) {
                // Streams have probably be closed externally (or broken)
                running = false;
            } catch (Throwable t) {
                // Something went wrong during execution
                // Store the exception and display a minimal error message
                session.put(Constants.EXCEPTION_VARIABLE, t);
                printError(t.getClass().getSimpleName(), t.getMessage());
            }
        }

        // Exit callback to be invoked
        if (callback != null) {
            callback.run();
        }

        // Session is not closed in JLineConsole.close() because if so, the
        // callback could not invoke any cleanup command
        session.close();

    }

    private void doSessionBranding() {
        // 1. Print the banner
        session.getConsole().println(brandingService.getBanner(reader.getTerminal().isAnsiSupported()));

        // 2. Inject variables
        for (Map.Entry<String, Object> entry : brandingService.getVariables().entrySet()) {
            session.put(entry.getKey(), entry.getValue());
        }

        // 3. Execute customization script
        if (brandingService.getScript() != null) {
            for (String line : brandingService.getScript()) {
                try {
                    Object result = session.execute(line);
                    session.put(Constants.LAST_RESULT_VARIABLE, result);
                    if (result != null) {
                        session.getConsole().println(session.format(result, Converter.INSPECT));
                    }
                } catch (Exception e) {
                    printError(e.getClass().getSimpleName(), e.getMessage());
                }
            }
        }
    }

    private String getRenderedPrompt() {
        String prompt = promptService.getPrompt(sessionVariables);
        if (AnsiRenderer.test(prompt)) {
            prompt = AnsiRenderer.render(prompt);
        }
        return prompt;
    }


    private void printError(String type, String message) {
        String error = Ansi.ansi().fg(Ansi.Color.RED)
                           .a("[")
                           .a(type)
                           .a("]: ")
                           .reset().toString();
        session.getConsole().println(error + message);
    }

    private class SessionVariables implements Variables {

        public Object get(String name) {
            return session.get(name);
        }
    }

}

