/**
 * Copyright 2012 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.internal.converter;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.service.command.Converter;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Converts a {@code Number} instance into another {@code Number} subtype.
 * @author Loris Bouzonnet
 */
@Component
@Provides
public class NumberConverter implements Converter {

    @ServiceProperty(name = Converter.CONVERTER_CLASSES,
                     value = "java.lang.Number")
    private String supportedClasses;

    public Object convert(Class<?> toType, Object in) throws Exception {
        if (in instanceof Number && Number.class.isAssignableFrom(toType)) {
            return convertToNumber((Number) in, toType);
        }
        return null;
    }

    private Object convertToNumber(Number value, Class toType) throws Exception {
        if (AtomicInteger.class.equals(toType)) {
            // Convert to Integer first
            return new AtomicInteger((Integer) convertToNumber(value, Integer.class));
        } else if (AtomicLong.class.equals(toType)) {
            // Convert to Long first
            return new AtomicLong((Long) convertToNumber(value, Long.class));
        } else if (Integer.class.equals(toType)) {
            return value.intValue();
        } else if (Short.class.equals(toType)) {
            return value.shortValue();
        } else if (Long.class.equals(toType)) {
            return value.longValue();
        } else if (Float.class.equals(toType)) {
            return value.floatValue();
        } else if (Double.class.equals(toType)) {
            return value.doubleValue();
        } else if (Byte.class.equals(toType)) {
            return value.byteValue();
        } else if (BigInteger.class.equals(toType)) {
            return new BigInteger(value.toString());
        } else if (BigDecimal.class.equals(toType)) {
            return new BigDecimal(value.toString());
        } else {
            // No other Number sub-types
            return null;
        }
    }

    public CharSequence format(Object o, int level, Converter converter) throws Exception {
        if (o instanceof Number) {
            return o.toString();
        }
        return null;
    }

}
