/**
 * Copyright 2013 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.internal.handler.completer;

import jline.console.ConsoleReader;
import jline.console.CursorBuffer;
import jline.console.completer.CandidateListCompletionHandler;

import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Override the method {@link CandidateListCompletionHandler#getUnambiguousCompletions(java.util.List)} to be aware of scopes.
 */
public class ShelbieCandidateListCompletionHandler extends CandidateListCompletionHandler {

    @Override
    public boolean complete(final ConsoleReader reader, final List<CharSequence> candidates, final int pos) throws
            IOException
    {
        CursorBuffer buf = reader.getCursorBuffer();

        // if there is only one completion, then fill in the buffer
        if (candidates.size() == 1) {
            CharSequence value = candidates.get(0);

            // fail if the only candidate is the same as the current buffer
            if (value.equals(buf.toString())) {
                return false;
            }

            setBuffer(reader, value, pos);

            return true;
        }
        else if (candidates.size() > 1) {
            String value = getUnambiguousCompletions(buf, candidates);
            setBuffer(reader, value, pos);
        }

        printCandidates(reader, candidates);

        // redraw the current console buffer
        reader.drawLine();

        return true;
    }

    /**
     * Returns a root that matches all the {@link String} elements of the specified {@link List},
     * or null if there are no commonalities. For example, if the list contains
     * <i>foobar</i>, <i>foobaz</i>, <i>foobuz</i>, the method will return <i>foob</i>.
     */
    private String getUnambiguousCompletions(final CursorBuffer buf, final List<CharSequence> candidates) {
        if (candidates == null || candidates.isEmpty()) {
            return null;
        }

        String[] strings;

        String buffer = buf.toString();
        if (buffer.contains(":")) {
            strings = candidates.toArray(new String[candidates.size()]);
        } else {
            Set<String> scopeAndCommands = new HashSet<String>();
            for (final CharSequence c : candidates) {
                String[] scopeAndCommand = c.toString().split(":");
                if (scopeAndCommand[0].startsWith(buffer)) {
                    scopeAndCommands.add(scopeAndCommand[0]);
                }
                if (scopeAndCommand[1].startsWith(buffer)) {
                    scopeAndCommands.add(scopeAndCommand[1]);
                }
            }
            strings = scopeAndCommands.toArray(new String[scopeAndCommands.size()]);
        }

        StringBuilder candidate = new StringBuilder();

        for (int i = 0; i < strings[0].length(); i++) {
            if (startsWith(strings[0].substring(0, i + 1), strings)) {
                candidate.append(strings[0].charAt(i));
            }
            else {
                break;
            }
        }

        return candidate.toString();
    }

    /**
     * @return true is all the elements of <i>candidates</i> start with <i>starts</i>
     */
    private boolean startsWith(final String starts, final String[] candidates) {
        for (String candidate : candidates) {
            if (!candidate.startsWith(starts)) {
                return false;
            }
        }

        return true;
    }
}
