/**
 * Copyright 2013 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.internal.prompt;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Property;
import org.apache.felix.ipojo.annotations.Provides;
import org.ow2.shelbie.core.console.Constants;
import org.ow2.shelbie.core.prompt.PromptService;
import org.ow2.shelbie.core.prompt.Variables;
import org.ow2.util.substitution.IPropertyResolver;
import org.ow2.util.substitution.ISubstitutionEngine;
import org.ow2.util.substitution.engine.DefaultSubstitutionEngine;
import org.ow2.util.substitution.resolver.ChainedResolver;
import org.ow2.util.substitution.resolver.FallbackResolver;
import org.ow2.util.substitution.resolver.PropertiesResolver;
import org.ow2.util.substitution.resolver.RecursiveResolver;

/**
 * The default prompt supports expression resolution.
 * It is initially configured with a default prompt template (but can be overridden).
 * It also supports prompt configuration based on session variable (variable
 * {@literal prompt} to be used just like {@literal PS1} on Unix systems).
 */
@Component
@Provides
public class DefaultPromptService implements PromptService {
    /**
     * Default prompt.
     * TODO Use Ansi coloring
     */
    private static final String DEFAULT_PROMPT = "${user.name}@${application.name}$ ";

    /**
     * Prompt to be used when in case of error.
     */
    private static final String ERROR_PROMPT = "[shelbie]$ ";

    /**
     * Substitution engine for prompt resolution
     */
    private ISubstitutionEngine engine;

    private IPropertyResolver resolver;

    @Property(value = DEFAULT_PROMPT)
    private String template = DEFAULT_PROMPT;

    public DefaultPromptService() {
        engine = createEngine();
    }

    public String getPrompt(final Variables variables) {

        String prompt = (String) variables.get(Constants.PROMPT_VARIABLE);
        if (prompt == null) {
            prompt = template;
        }

        // Resolve the prompt
        try {
            resolver = new VariablesResolver(variables);
            return engine.substitute(prompt);
        } catch (Throwable t) {
            return ERROR_PROMPT;
        }
    }

    /**
     * Create a dedicated substitution engine
     *
     * @return a configured SubstitutionEngine
     */
    private ISubstitutionEngine createEngine() {

        // Configure the substitution engine
        DefaultSubstitutionEngine engine = new DefaultSubstitutionEngine();
        engine.setMarkerChar('$');
        engine.setOpeningChar('{');
        engine.setEndingChar('}');

        // Chain all required resolvers
        ChainedResolver chained = new ChainedResolver();
        chained.getResolvers().add(new DelegateResolver());
        chained.getResolvers().add(new PropertiesResolver(System.getProperties()));
        chained.getResolvers().add(new FallbackResolver());

        // Enable recursive resolution
        RecursiveResolver recursive = new RecursiveResolver(engine, chained);
        engine.setResolver(recursive);

        return engine;
    }

    private class DelegateResolver implements IPropertyResolver {

        public String resolve(String expression) {
            if (resolver == null) {
                return null;
            }
            return resolver.resolve(expression);
        }
    }

}
