/**
 * Copyright 2013 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.internal.registry;

import org.apache.felix.ipojo.annotations.Bind;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Unbind;
import org.ow2.shelbie.core.registry.CommandFilter;
import org.ow2.shelbie.core.registry.CommandRegistry;
import org.ow2.shelbie.core.registry.info.CommandInfo;
import org.ow2.shelbie.core.registry.info.ScopeInfo;
import org.ow2.shelbie.core.registry.supply.InfoSupplier;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * User: guillaume
 * Date: 14/02/13
 * Time: 13:55
 */
@Component
@Provides
public class DefaultCommandRegistry implements CommandRegistry {

    private List<InfoSupplier> suppliers = new ArrayList<InfoSupplier>();

    public Collection<? extends ScopeInfo> getScopes() {
        Map<String, AggregatedScopeInfo> scopes = new HashMap<String, AggregatedScopeInfo>();

        for (InfoSupplier supplier : suppliers) {
            Collection<ScopeInfo> supplied = supplier.getScopeInfo();
            for (ScopeInfo scopeInfo : supplied) {
                AggregatedScopeInfo asi = findScopeInfo(scopes, scopeInfo.getName());
                asi.addScopeInfo(scopeInfo);
            }
        }

        return scopes.values();
    }

    public Collection<? extends CommandInfo> getAllCommands() {
        Collection<CommandInfo> commands = new ArrayList<CommandInfo>();
        for (ScopeInfo scopeInfo : getScopes()) {
            commands.addAll(scopeInfo.getCommands());
        }
        return commands;
    }

    public Collection<? extends CommandInfo> getCommands(CommandFilter filter) {
        Collection<CommandInfo> commands = new ArrayList<CommandInfo>();
        for (CommandInfo commandInfo : getAllCommands()) {
            if (filter.accept(commandInfo)) {
                commands.add(commandInfo);
            }
        }
        return commands;
    }

    @Bind(aggregate = true, optional = true)
    public void bindInfoSupplier(InfoSupplier supplier) {
        suppliers.add(supplier);
    }

    @Unbind
    public void unbindInfoSupplier(InfoSupplier supplier) {
        suppliers.remove(supplier);
    }

    private AggregatedScopeInfo findScopeInfo(Map<String, AggregatedScopeInfo> scopes, String scope) {
        AggregatedScopeInfo asi = scopes.get(scope);
        if (asi == null) {
            asi = new AggregatedScopeInfo(scope);
            scopes.put(scope, asi);
        }
        return asi;
    }

}
