/**
 * Copyright 2012-2013 Peergreen S.A.S.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.console.startup;

import java.io.File;
import java.security.PrivilegedAction;

import javax.security.auth.Subject;

import jline.TerminalFactory;
import jline.console.completer.Completer;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.Validate;
import org.apache.felix.service.command.CommandProcessor;
import org.ow2.shelbie.core.branding.BrandingService;
import org.ow2.shelbie.core.console.JLineConsole;
import org.ow2.shelbie.core.console.OperatingSystem;
import org.ow2.shelbie.core.history.HistoryFileProvider;
import org.ow2.shelbie.core.internal.security.RolesPrincipal;
import org.ow2.shelbie.core.internal.security.UserPrincipal;
import org.ow2.shelbie.core.prompt.PromptService;
import org.ow2.shelbie.core.system.SystemService;

/**
 * Created by IntelliJ IDEA.
 * User: sauthieg
 * Date: 6 janv. 2010
 * Time: 16:44:07
 * To change this template use File | Settings | File Templates.
 */
@Component
public class ConsoleStartupComponent {

    @Requires
    private CommandProcessor processor;

    @Requires
    private SystemService system;

    @Requires(filter = "(type=commands)")
    private Completer completer;

    @Requires(policy = "dynamic-priority")
    private PromptService promptService;

    @Requires(policy = "dynamic-priority",
              optional = true,
              defaultimplementation = NullHistoryFileProvider.class)
    private HistoryFileProvider historyProvider;

    @Requires(optional = true, nullable = false, proxy = false)
    private BrandingService brandingService;

    private JLineConsole console;

    @Validate
    public void startup() throws Exception {

        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(ConsoleStartupComponent.class.getClassLoader());

        try {
            // Start the console
            console = new JLineConsole(processor,
                                       completer,
                                       system.getIn(),
                                       system.getOut(),
                                       system.getErr(),
                                       TerminalFactory.get());

            // This is a local console, when closed, the system must shutdown automatically
            console.setCallback(new ShutdownCallback());

            Subject subject = createLocalSubject();
            console.setHistoryFile(historyProvider.getHistoryFile(subject));
            console.setPromptService(promptService);
            console.setBrandingService(brandingService);

            Thread thread = new Thread(new PrivilegedRunnable(subject, console), "Shelbie Console Thread");
            thread.setDaemon(true);
            thread.start();

            // Store some global properties
            console.getSession().put("application.name", "shelbie");
            console.getSession().put(Subject.class.getName(), subject);

            // TODO, handle property substitution using Beans
            OperatingSystem os = new OperatingSystem();
            os.setName(System.getProperty("os.name"));
            os.setArch(System.getProperty("os.arch"));
            os.setVersion(System.getProperty("os.version"));
            console.getSession().put("os", os);
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }
    }

    private Subject createLocalSubject() {
        Subject subject = new Subject();
        subject.getPrincipals().add(new UserPrincipal(System.getProperty("user.name")));
        subject.getPrincipals().add(new RolesPrincipal("admin"));
        return subject;
    }

    @Invalidate
    public void shutdown() {
        console.close();
    }

    public static class NullHistoryFileProvider implements HistoryFileProvider {
        public File getHistoryFile(Subject subject) {
            return null;
        }
    }

    private static class PrivilegedRunnable implements Runnable {
        private final Subject subject;
        private final Runnable delegate;

        protected PrivilegedRunnable(Subject subject, Runnable delegate) {
            this.subject = subject;
            this.delegate = delegate;
        }

        public void run() {
            Subject.doAs(subject, new PrivilegedAction<Object>() {
                public Object run() {
                    delegate.run();
                    return true;
                }
            });
        }
    }

    private class ShutdownCallback implements Runnable {
        public void run() {
            if (!system.isStopping()) {
                try {
                    // This is called after a graceful shell exit
                    console.getSession().execute("shelbie:shutdown --force --quiet");
                } catch (Exception e) {
                    // Ignore
                }
            }
        }
    }
}
