/**
 * Copyright 2013 Peergreen SAS
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.console.util;

import org.fusesource.jansi.AnsiConsole;

import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Method;

/**
 * the {@literal Streams} utility class holds streams related stuff.
 */
public class Streams {

    private Streams() {}

    /**
     * Wrap the given {@link PrintStream} into an Ansi capable {@link PrintStream}.
     * @param stream stream to be wrapped
     * @return ANSI capable stream
     */
    public static PrintStream wrap(PrintStream stream) {

        // Try top unwrap the given Stream to avoid infinite recursion
        PrintStream ps = unwrapStream(stream);

        // Enable ANSI support on the stream
        OutputStream result = AnsiConsole.wrapOutputStream(ps);

        // Return a PrintStream instance
        if (result instanceof PrintStream) {
            return (PrintStream) result;
        }
        return new PrintStream(result);
    }

    /**
     * The {@link System#out} {@link PrintStream} provided by Gogo is itself a wrapper to the real interesting PrintStream.
     * If we don't extract that stream, we'll hit some infinite recursion issue, and that's not cool.
     * @param stream Potential Gogo stream
     * @return unwrapped stream
     */
    public static PrintStream unwrapStream(PrintStream stream) {
        try {
            // Used reflection to avoid a dependency on gogo ThreadPrintStream class
            Method method = stream.getClass().getMethod("getRoot");
            return (PrintStream) method.invoke(stream);
        } catch (Throwable t) {
            // Could not unwrap the stream, return it unchanged
            return stream;
        }
    }

}
