/**
 * Copyright 2013 Peergreen SAS
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.internal.history;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.ow2.shelbie.core.identity.IdentityProvider;

import javax.security.auth.Subject;
import java.security.Principal;
import java.security.acl.Group;
import java.util.HashSet;
import java.util.Set;

/**
 * The default {@link IdentityProvider} implements a generic extraction strategy:
 * If the {@link Subject} contains only 1 {@link Principal} (not a {@link Group}),
 * it will return that Principal's name.
 * It will throw an Exception if there is not only one mono-valued Principal.
 */
@Component
@Provides
public class DefaultIdentityProvider implements IdentityProvider {

    public String get(Subject subject) {
        Set<Principal> filtered = new HashSet<Principal>();
        Set<Principal> principals = subject.getPrincipals();
        for (Principal principal : principals) {
            if (!(principal instanceof Group)) {
                filtered.add(principal);
            }
        }

        if (filtered.isEmpty()) {
            throw new IllegalStateException("Subject does not contain any mono-valued Principals");
        }

        if (filtered.size() > 1) {
            // TODO maybe we can relax a bit and return the first one
            throw new IllegalStateException("Subject does not contain only 1 mono-valued Principal (found:" + filtered.size() + ").");
        }
        return filtered.iterator().next().getName();
    }
}
