/**
 * Copyright 2013 Peergreen SAS
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.internal.registry.supplier.gogo.info;

import org.apache.felix.service.command.CommandSession;
import org.apache.felix.service.command.Descriptor;
import org.apache.felix.service.command.Parameter;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.ow2.shelbie.core.internal.registry.util.References;
import org.ow2.shelbie.core.registry.info.ScopeInfo;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * User: guillaume
 * Date: 15/02/13
 * Time: 13:37
 */
public class InfoBuilder {

    // This is part of the Gogo internals (Reflective.java)
    public final static Set<String> KEYWORDS = new HashSet<String>(
            Arrays.asList(new String[]{"abstract", "continue", "for", "new", "switch",
                    "assert", "default", "goto", "package", "synchronized", "boolean", "do",
                    "if", "private", "this", "break", "double", "implements", "protected",
                    "throw", "byte", "else", "import", "public", "throws", "case", "enum",
                    "instanceof", "return", "transient", "catch", "extends", "int", "short",
                    "try", "char", "final", "interface", "static", "void", "class",
                    "finally", "long", "strictfp", "volatile", "const", "float", "native",
                    "super", "while"}));

    private final BundleContext bundleContext;

    public InfoBuilder(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    public ScopeInfo buildScopeInfo(ServiceReference reference) {

        GogoScopeInfo scopeInfo;
        try {
            Object support = bundleContext.getService(reference);
            scopeInfo = new GogoScopeInfo(References.getCommandScope(reference));

            List<String> names = References.getCommandFunctions(reference);
            // Only public methods may be used as commands
            List<Method> methods = Arrays.asList(support.getClass().getMethods());

            for (String name : names) {

                String unmodifiedName = name;

                // These reserved keywords cannot be method names
                if (KEYWORDS.contains(name)) {
                    name = "_" + name;
                }

                for (Method method : findMethods(methods, name)) {
                    GogoCommandInfo commandInfo = new GogoCommandInfo(scopeInfo.getName(), unmodifiedName);
                    scopeInfo.getCommands().add(commandInfo);

                    Descriptor md = method.getAnnotation(Descriptor.class);
                    if (md != null) {
                        commandInfo.setDescription(md.value());
                    }

                    int argumentIndex = 0;
                    for (int i = 0; i < method.getParameterTypes().length; i++) {

                        // Ignore CommandSession first parameter
                        if (!method.getParameterTypes()[i].equals(CommandSession.class)) {
                            Annotation[] annotations = method.getParameterAnnotations()[i];
                            Parameter parameter = getAnnotation(annotations, Parameter.class);
                            if (parameter != null) {
                                GogoOptionInfo optionInfo = new GogoOptionInfo(parameter.names(), method.getParameterTypes()[i]);
                                commandInfo.getOptions().add(optionInfo);
                                optionInfo.setDefaultValue(parameter.absentValue());

                                Descriptor od = getAnnotation(annotations, Descriptor.class);
                                if (od != null) {
                                    optionInfo.setDescription(od.value());
                                }
                            } else {
                                // It's an argument
                                GogoArgumentInfo argumentInfo = new GogoArgumentInfo(method.getParameterTypes()[i], argumentIndex++);
                                commandInfo.getArguments().add(argumentInfo);

                                Descriptor od = getAnnotation(annotations, Descriptor.class);
                                if (od != null) {
                                    argumentInfo.setDescription(od.value());
                                }

                            }
                        }
                    }
                }

            }
        } finally {
            bundleContext.ungetService(reference);
        }

        return scopeInfo;
    }

    private <T> T getAnnotation(Annotation[] annotations, Class<T> annotationType) {
        for (Annotation annotation : annotations) {
            if (annotationType.equals(annotation.annotationType())) {
                return annotationType.cast(annotation);
            }
        }
        return null;
    }

    private List<Method> findMethods(List<Method> methods, String name) {
        List<Method> filtered = new ArrayList<Method>();
        for (Method method : methods) {
            if (method.getName().equals(name)) {
                filtered.add(method);
            }
        }

        if (filtered.isEmpty()) {
            throw new IllegalStateException("Cannot find matching method for function named " + name);
        }

        return filtered;
    }

}
